<?php
require_once '../../includes/middleware.php';
requirePermission('create_course');
require_once '../../config/database.php';
require_once '../../config/currency.php';
require_once '../../includes/functions.php';

$db = getDB();

// Get categories for dropdown
$stmt = $db->query("SELECT * FROM course_categories WHERE is_active = 1 ORDER BY name");
$categories = $stmt->fetchAll();

$message = '';
$error = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $title = trim($_POST['title']);
    $slug = trim($_POST['slug'] ?? '');
    $description = trim($_POST['description']);
    $shortDescription = trim($_POST['short_description']);
    $categoryId = !empty($_POST['category_id']) ? $_POST['category_id'] : null;
    $level = $_POST['level'];
    $language = trim($_POST['language']);
    $durationHours = intval($_POST['duration_hours']);
    $price = floatval($_POST['price']);
    $currency = trim($_POST['currency']);
    $maxStudents = !empty($_POST['max_students']) ? intval($_POST['max_students']) : null;
    $prerequisites = trim($_POST['prerequisites']);
    $learningObjectives = array_filter(array_map('trim', $_POST['learning_objectives'] ?? []));
    $tags = array_filter(array_map('trim', $_POST['tags'] ?? []));
    $isFree = isset($_POST['is_free']);
    $isFeatured = isset($_POST['is_featured']);
    $status = $_POST['status'];

    // Auto-generate slug if empty
    if (empty($slug)) {
        $baseSlug = strtolower(preg_replace('/[^a-zA-Z0-9\-]/', '', str_replace(' ', '-', $title)));
        $slug = $baseSlug;
        $counter = 1;

        // Check if slug exists and append number if needed
        while (true) {
            $stmt = $db->prepare("SELECT COUNT(*) FROM courses WHERE slug = ?");
            $stmt->execute([$slug]);
            $count = $stmt->fetchColumn();

            if ($count == 0) {
                break; // Slug is unique
            }

            $slug = $baseSlug . '-' . $counter;
            $counter++;

            // Prevent infinite loop
            if ($counter > 100) {
                $slug = $baseSlug . '-' . time();
                break;
            }
        }
    }

    // Validate required fields
    if (empty($title) || empty($description)) {
        $error = 'Title and description are required.';
    } else {
        try {
            // Handle file uploads
            $thumbnail = '';
            $bannerImage = '';

            if (isset($_FILES['thumbnail']) && $_FILES['thumbnail']['error'] === UPLOAD_ERR_OK) {
                $uploadDir = '../../uploads/course-thumbnails/';
                if (!is_dir($uploadDir)) mkdir($uploadDir, 0755, true);

                $fileName = time() . '_' . basename($_FILES['thumbnail']['name']);
                $targetPath = $uploadDir . $fileName;

                if (move_uploaded_file($_FILES['thumbnail']['tmp_name'], $targetPath)) {
                    $thumbnail = 'uploads/course-thumbnails/' . $fileName;
                }
            }

            if (isset($_FILES['banner_image']) && $_FILES['banner_image']['error'] === UPLOAD_ERR_OK) {
                $uploadDir = '../../uploads/course-banners/';
                if (!is_dir($uploadDir)) mkdir($uploadDir, 0755, true);

                $fileName = time() . '_' . basename($_FILES['banner_image']['name']);
                $targetPath = $uploadDir . $fileName;

                if (move_uploaded_file($_FILES['banner_image']['tmp_name'], $targetPath)) {
                    $bannerImage = 'uploads/course-banners/' . $fileName;
                }
            }

            // Insert course
            $stmt = $db->prepare("INSERT INTO courses (title, slug, description, short_description, instructor_id, category_id, thumbnail, banner_image, price, currency, level, duration_hours, language, max_students, prerequisites, learning_objectives, tags, status, is_featured, is_free, created_at, updated_at) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW(), NOW())");

            $stmt->execute([
                $title, $slug, $description, $shortDescription, $_SESSION['user_id'], $categoryId,
                $thumbnail, $bannerImage, $price, $currency, $level, $durationHours, $language,
                $maxStudents, $prerequisites, json_encode($learningObjectives), json_encode($tags),
                $status, $isFeatured, $isFree
            ]);

            $courseId = $db->lastInsertId();

            $message = 'Course created successfully! <a href="edit.php?id=' . $courseId . '" class="text-blue-600 hover:text-blue-800">Add curriculum and content</a>';

        } catch (PDOException $e) {
            $error = 'Error creating course: ' . $e->getMessage();
        }
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Create New Course - Mutalex Academy</title>
    <!-- Load Tailwind CSS CDN -->
    <script src="https://cdn.tailwindcss.com"></script>
    <!-- Font Awesome for icons -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css">
</head>
<body class="bg-gray-50 min-h-screen">

<div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
    <div class="bg-white shadow-lg rounded-lg p-6">
        <div class="flex justify-between items-center mb-6">
            <h1 class="text-3xl font-bold text-gray-900">Create New Course</h1>
            <a href="../dashboard.php" class="bg-gray-600 text-white px-4 py-2 rounded-lg hover:bg-gray-700 transition">
                <i class="fas fa-arrow-left mr-2"></i>Back to Dashboard
            </a>
        </div>

        <!-- Alerts -->
        <?php if ($message): ?>
            <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded mb-4">
                <?php echo $message; ?>
            </div>
        <?php endif; ?>
        <?php if ($error): ?>
            <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-4">
                <?php echo htmlspecialchars($error); ?>
            </div>
        <?php endif; ?>

        <form method="POST" enctype="multipart/form-data" class="space-y-6">
            <!-- Basic Information -->
            <div class="bg-gray-50 p-4 rounded-lg">
                <h2 class="text-xl font-semibold text-gray-800 mb-4">Basic Information</h2>
                <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <div>
                        <label for="title" class="block text-sm font-medium text-gray-700 mb-1">Course Title *</label>
                        <input type="text" class="w-full border border-gray-300 rounded-md px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500" id="title" name="title" required>
                    </div>
                    <div>
                        <label for="slug" class="block text-sm font-medium text-gray-700 mb-1">URL Slug</label>
                        <input type="text" class="w-full border border-gray-300 rounded-md px-3 py-2" id="slug" name="slug" placeholder="auto-generated-from-title">
                    </div>
                </div>
                <div class="mt-4">
                    <label for="shortDescription" class="block text-sm font-medium text-gray-700 mb-1">Short Description</label>
                    <textarea class="w-full border border-gray-300 rounded-md px-3 py-2" id="shortDescription" name="short_description" rows="3"></textarea>
                </div>
                <div class="mt-4">
                    <label for="description" class="block text-sm font-medium text-gray-700 mb-1">Full Description *</label>
                    <textarea class="w-full border border-gray-300 rounded-md px-3 py-2" id="description" name="description" rows="6" required></textarea>
                </div>
            </div>

            <!-- Course Details -->
            <div class="bg-gray-50 p-4 rounded-lg">
                <h2 class="text-xl font-semibold text-gray-800 mb-4">Course Details</h2>
                <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <div>
                        <label for="category" class="block text-sm font-medium text-gray-700 mb-1">Category</label>
                        <select class="w-full border border-gray-300 rounded-md px-3 py-2" id="category" name="category_id">
                            <option value="">Select Category</option>
                            <?php foreach ($categories as $category): ?>
                                <option value="<?php echo $category['id']; ?>"><?php echo htmlspecialchars($category['name']); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div>
                        <label for="level" class="block text-sm font-medium text-gray-700 mb-1">Difficulty Level</label>
                        <select class="w-full border border-gray-300 rounded-md px-3 py-2" id="level" name="level" required>
                            <option value="beginner">Beginner</option>
                            <option value="intermediate">Intermediate</option>
                            <option value="advanced">Advanced</option>
                        </select>
                    </div>
                    <div>
                        <label for="language" class="block text-sm font-medium text-gray-700 mb-1">Language</label>
                        <input type="text" class="w-full border border-gray-300 rounded-md px-3 py-2" id="language" name="language" value="English" required>
                    </div>
                    <div>
                        <label for="duration" class="block text-sm font-medium text-gray-700 mb-1">Duration (hours)</label>
                        <input type="number" class="w-full border border-gray-300 rounded-md px-3 py-2" id="duration" name="duration_hours" min="1" value="10" required>
                    </div>
                </div>
                <div class="mt-4">
                    <label for="prerequisites" class="block text-sm font-medium text-gray-700 mb-1">Prerequisites</label>
                    <textarea class="w-full border border-gray-300 rounded-md px-3 py-2" id="prerequisites" name="prerequisites" rows="3"></textarea>
                </div>
                <!-- Learning Objectives -->
                <div class="mt-4">
                    <label class="block text-sm font-medium text-gray-700 mb-1">Learning Objectives</label>
                    <div id="objectives-container">
                        <div class="flex gap-2 mb-2">
                            <input type="text" class="flex-1 border border-gray-300 rounded-md px-3 py-2" name="learning_objectives[]" placeholder="What will students learn?">
                            <button type="button" class="bg-red-500 text-white px-3 py-2 rounded-md hidden remove-objective">
                                <i class="fas fa-trash"></i>
                            </button>
                        </div>
                    </div>
                    <button type="button" class="bg-blue-500 text-white px-4 py-2 rounded-md mt-2" id="add-objective">
                        <i class="fas fa-plus mr-1"></i>Add Objective
                    </button>
                </div>
                <!-- Tags -->
                <div class="mt-4">
                    <label class="block text-sm font-medium text-gray-700 mb-1">Tags</label>
                    <input type="text" class="w-full border border-gray-300 rounded-md px-3 py-2" name="tags_input" id="tags-input" placeholder="Enter tags separated by commas">
                    <div id="tags-container" class="mt-2"></div>
                </div>
            </div>

            <!-- Pricing & Settings -->
            <div class="bg-gray-50 p-4 rounded-lg">
                <h2 class="text-xl font-semibold text-gray-800 mb-4">Pricing & Settings</h2>
                <div class="flex items-center mb-4">
                    <input type="checkbox" class="mr-2" id="isFree" name="is_free" checked>
                    <label for="isFree" class="text-sm font-medium text-gray-700">Free Course</label>
                </div>
                <div id="pricing-fields" class="hidden">
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <div>
                            <label for="price" class="block text-sm font-medium text-gray-700 mb-1">Price</label>
                            <div class="flex">
                                <span class="inline-flex items-center px-3 py-2 border border-r-0 border-gray-300 bg-gray-50 text-gray-500 text-sm rounded-l-md">$</span>
                                <input type="number" class="flex-1 border border-gray-300 rounded-r-md px-3 py-2" id="price" name="price" min="0" step="0.01" value="0.00">
                            </div>
                        </div>
                        <div>
                            <label for="currency" class="block text-sm font-medium text-gray-700 mb-1">Currency</label>
                            <select class="w-full border border-gray-300 rounded-md px-3 py-2" id="currency" name="currency">
                                <option value="USD">USD</option>
                                <option value="EUR">EUR</option>
                            </select>
                        </div>
                    </div>
                </div>
                <div class="mt-4">
                    <label for="maxStudents" class="block text-sm font-medium text-gray-700 mb-1">Maximum Students</label>
                    <input type="number" class="w-full border border-gray-300 rounded-md px-3 py-2" id="maxStudents" name="max_students" min="1" placeholder="Unlimited">
                </div>
                <div class="mt-4">
                    <label for="status" class="block text-sm font-medium text-gray-700 mb-1">Status</label>
                    <select class="w-full border border-gray-300 rounded-md px-3 py-2" id="status" name="status" required>
                        <option value="draft">Draft (not visible to students)</option>
                        <option value="published">Published (visible to students)</option>
                    </select>
                </div>
                <div class="flex items-center mt-4">
                    <input type="checkbox" class="mr-2" id="isFeatured" name="is_featured">
                    <label for="isFeatured" class="text-sm font-medium text-gray-700">Featured Course</label>
                </div>
            </div>

            <!-- Media -->
            <div class="bg-gray-50 p-4 rounded-lg">
                <h2 class="text-xl font-semibold text-gray-800 mb-4">Media</h2>
                <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <div>
                        <label for="thumbnail" class="block text-sm font-medium text-gray-700 mb-1">Course Thumbnail</label>
                        <input type="file" class="w-full border border-gray-300 rounded-md px-3 py-2" id="thumbnail" name="thumbnail" accept="image/*">
                        <p class="text-xs text-gray-500 mt-1">Recommended: 1280x720px, JPG/PNG</p>
                    </div>
                    <div>
                        <label for="bannerImage" class="block text-sm font-medium text-gray-700 mb-1">Banner Image</label>
                        <input type="file" class="w-full border border-gray-300 rounded-md px-3 py-2" id="bannerImage" name="banner_image" accept="image/*">
                        <p class="text-xs text-gray-500 mt-1">Optional banner for course header</p>
                    </div>
                </div>
            </div>

            <div class="flex justify-end">
                <button type="submit" class="bg-blue-600 text-white px-6 py-3 rounded-lg hover:bg-blue-700 transition font-medium">
                    <i class="fas fa-save mr-2"></i>Create Course
                </button>
            </div>
        </form>
    </div>
</div>

<script>
// Auto-generate slug
document.getElementById('title').addEventListener('input', function() {
    const title = this.value;
    const baseSlug = title.toLowerCase().replace(/[^a-z0-9\s-]/g, '').replace(/\s+/g, '-').replace(/-+/g, '-').trim('-');
    // Note: PHP will handle uniqueness on the server side
    document.getElementById('slug').value = baseSlug;
});

// Toggle pricing
document.getElementById('isFree').addEventListener('change', function() {
    document.getElementById('pricing-fields').classList.toggle('hidden', this.checked);
});

// Objectives
document.getElementById('add-objective').addEventListener('click', function() {
    const container = document.getElementById('objectives-container');
    const div = document.createElement('div');
    div.className = 'flex gap-2 mb-2';
    div.innerHTML = '<input type="text" class="flex-1 border border-gray-300 rounded-md px-3 py-2" name="learning_objectives[]" placeholder="What will students learn?"><button type="button" class="bg-red-500 text-white px-3 py-2 rounded-md remove-objective"><i class="fas fa-trash"></i></button>';
    container.appendChild(div);
    updateRemoveButtons();
});

document.getElementById('objectives-container').addEventListener('click', function(e) {
    if (e.target.classList.contains('remove-objective') || e.target.parentElement.classList.contains('remove-objective')) {
        e.target.closest('.flex').remove();
        updateRemoveButtons();
    }
});

function updateRemoveButtons() {
    const objectives = document.querySelectorAll('#objectives-container .flex');
    objectives.forEach((obj, index) => {
        const btn = obj.querySelector('.remove-objective');
        btn.classList.toggle('hidden', objectives.length <= 1);
    });
}

// Tags
let tags = [];
document.getElementById('tags-input').addEventListener('keydown', function(e) {
    if (e.key === 'Enter' || e.key === ',') {
        e.preventDefault();
        const tag = this.value.trim();
        if (tag && !tags.includes(tag)) {
            tags.push(tag);
            updateTagsDisplay();
            this.value = '';
        }
    }
});

function updateTagsDisplay() {
    const container = document.getElementById('tags-container');
    container.innerHTML = '';
    tags.forEach((tag, index) => {
        const span = document.createElement('span');
        span.className = 'inline-block bg-blue-100 text-blue-800 px-2 py-1 rounded mr-2 mb-2';
        span.innerHTML = `${tag} <button type="button" class="ml-1 text-red-500" onclick="removeTag(${index})">&times;</button>`;
        container.appendChild(span);
    });
    // Hidden inputs
    tags.forEach(tag => {
        const input = document.createElement('input');
        input.type = 'hidden';
        input.name = 'tags[]';
        input.value = tag;
        container.appendChild(input);
    });
}

function removeTag(index) {
    tags.splice(index, 1);
    updateTagsDisplay();
}
</script>

</body>
</html>