<?php
require_once '../../config/database.php';
require_once '../../config/auth.php';
require_once '../../config/rbac.php';
require_once '../../includes/functions.php';
require_once '../../includes/middleware.php';

// Check if user is instructor or admin
anyRoleMiddleware(['instructor', 'admin']);

$user = getUserContext();
$db = getDB();
$userId = $_SESSION['user_id'];

// Handle filters and search
$search = $_GET['search'] ?? '';
$type_filter = $_GET['type'] ?? '';
$status_filter = $_GET['status'] ?? '';
$course_filter = $_GET['course'] ?? '';

// Build query
$query = "
    SELECT e.*, c.title as course_title,
           COUNT(eq.id) as question_count,
           COUNT(DISTINCT ea.student_id) as attempt_count
    FROM exams e
    LEFT JOIN courses c ON e.course_id = c.id
    LEFT JOIN exam_questions eq ON e.id = eq.exam_id
    LEFT JOIN exam_attempts ea ON e.id = ea.exam_id
    WHERE e.instructor_id = ?
";

$params = [$userId];

if (!empty($search)) {
    $query .= " AND (e.title LIKE ? OR e.description LIKE ?)";
    $params[] = "%$search%";
    $params[] = "%$search%";
}

if (!empty($type_filter)) {
    $query .= " AND e.exam_type = ?";
    $params[] = $type_filter;
}

if (!empty($status_filter)) {
    $query .= " AND e.status = ?";
    $params[] = $status_filter;
}

if (!empty($course_filter)) {
    $query .= " AND e.course_id = ?";
    $params[] = $course_filter;
}

$query .= " GROUP BY e.id ORDER BY e.created_at DESC";

// Get exams
$stmt = $db->prepare($query);
$stmt->execute($params);
$exams = $stmt->fetchAll();

// Get filter options
$courses = $db->prepare("SELECT id, title FROM courses WHERE instructor_id = ? ORDER BY title", [$userId])->fetchAll();

$page_title = 'Exams - Mutalex Academy';
?>

<div class="admin-container">
    <div class="admin-sidebar">
        <div class="sidebar-header">
            <h2><i class="fas fa-graduation-cap me-2"></i>Mutalex Academy</h2>
            <p>Instructor Panel</p>
        </div>

        <nav class="sidebar-nav">
            <ul>
                <li><a href="../dashboard.php"><i class="fas fa-tachometer-alt me-2"></i>Dashboard</a></li>
                <li><a href="../courses/"><i class="fas fa-book me-2"></i>My Courses</a></li>
                <li><a href="../courses/new.php"><i class="fas fa-plus me-2"></i>Create Course</a></li>
                <li><a href="../questions/index.php"><i class="fas fa-question-circle me-2"></i>Question Bank</a></li>
                <li><a href="index.php" class="active"><i class="fas fa-file-alt me-2"></i>Exams</a></li>
                <li><a href="../resources.php"><i class="fas fa-folder me-2"></i>Resources</a></li>
                <li><a href="../students.php"><i class="fas fa-users me-2"></i>Students</a></li>
                <li><a href="../assignments.php"><i class="fas fa-tasks me-2"></i>Assignments</a></li>
                <li><a href="/lms/instructor/grading/index.php"><i class="fas fa-chart-line me-2"></i>Grades</a></li>
                <li><a href="../../logout.php"><i class="fas fa-sign-out-alt me-2"></i>Logout</a></li>
            </ul>
        </nav>
    </div>

    <div class="admin-main">
        <div class="admin-header">
            <div class="d-flex justify-content-between align-items-center">
                <div>
                    <h1><i class="fas fa-file-alt text-primary me-3"></i>Exams</h1>
                    <p class="text-muted mb-0">Create and manage comprehensive assessments for your courses</p>
                </div>
                <div class="header-actions">
                    <a href="new.php" class="btn btn-primary btn-lg">
                        <i class="fas fa-plus me-2"></i>Create Exam
                    </a>
                    <a href="../questions/index.php" class="btn btn-success btn-lg">
                        <i class="fas fa-question-circle me-2"></i>Question Bank
                    </a>
                </div>
            </div>
        </div>

        <div class="admin-content">

            <!-- Filters -->
            <div class="filter-section">
                <div class="filter-form">
                    <div class="filter-row">
                        <div class="filter-group">
                            <label for="search"><i class="fas fa-search me-1"></i>Search</label>
                            <input type="text" id="search" class="form-control" name="search" placeholder="Search exams..."
                                   value="<?php echo htmlspecialchars($search); ?>">
                        </div>
                        <div class="filter-group">
                            <label for="type"><i class="fas fa-list me-1"></i>Exam Type</label>
                            <select class="form-select" id="type" name="type">
                                <option value="">All Types</option>
                                <option value="quiz" <?php echo $type_filter === 'quiz' ? 'selected' : ''; ?>>Quiz</option>
                                <option value="midterm" <?php echo $type_filter === 'midterm' ? 'selected' : ''; ?>>Midterm</option>
                                <option value="final" <?php echo $type_filter === 'final' ? 'selected' : ''; ?>>Final</option>
                                <option value="practice" <?php echo $type_filter === 'practice' ? 'selected' : ''; ?>>Practice</option>
                                <option value="certification" <?php echo $type_filter === 'certification' ? 'selected' : ''; ?>>Certification</option>
                            </select>
                        </div>
                        <div class="filter-group">
                            <label for="status"><i class="fas fa-info-circle me-1"></i>Status</label>
                            <select class="form-select" id="status" name="status">
                                <option value="">All Status</option>
                                <option value="draft" <?php echo $status_filter === 'draft' ? 'selected' : ''; ?>>Draft</option>
                                <option value="published" <?php echo $status_filter === 'published' ? 'selected' : ''; ?>>Published</option>
                                <option value="archived" <?php echo $status_filter === 'archived' ? 'selected' : ''; ?>>Archived</option>
                            </select>
                        </div>
                        <div class="filter-group">
                            <label for="course"><i class="fas fa-book me-1"></i>Course</label>
                            <select class="form-select" id="course" name="course">
                                <option value="">All Courses</option>
                                <?php foreach ($courses as $course): ?>
                                    <option value="<?php echo $course['id']; ?>" <?php echo $course_filter == $course['id'] ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($course['title']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="filter-actions">
                            <button type="submit" class="btn btn-primary">
                                <i class="fas fa-filter me-1"></i>Apply Filters
                            </button>
                            <a href="index.php" class="btn btn-outline-secondary">
                                <i class="fas fa-times me-1"></i>Clear
                            </a>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Exams List -->
            <div class="exams-grid-container">
                <div class="grid-header">
                    <h3><i class="fas fa-clipboard-list me-2"></i>Exams (<?php echo count($exams); ?>)</h3>
                </div>

                <?php if (empty($exams)): ?>
                    <div class="empty-state">
                        <div class="empty-state-icon">
                            <i class="fas fa-clipboard-list"></i>
                        </div>
                        <h3>No Exams Found</h3>
                        <p>Start creating assessments to evaluate your students' knowledge and skills.</p>
                        <a href="new.php" class="btn btn-primary btn-lg">
                            <i class="fas fa-plus me-2"></i>Create Your First Exam
                        </a>
                    </div>
                <?php else: ?>
                    <div class="exams-grid">
                        <?php foreach ($exams as $exam): ?>
                            <div class="exam-card">
                                <div class="exam-card-header">
                                    <div class="exam-title-section">
                                        <h4><?php echo htmlspecialchars($exam['title']); ?></h4>
                                        <span class="exam-type-badge <?php echo $exam['exam_type']; ?>">
                                            <?php echo ucfirst($exam['exam_type']); ?>
                                        </span>
                                    </div>
                                    <div class="exam-status-section">
                                        <span class="status-badge <?php
                                            echo $exam['status'] === 'published' ? 'active' :
                                                 ($exam['status'] === 'draft' ? 'draft' : 'archived');
                                        ?>">
                                            <?php echo ucfirst($exam['status']); ?>
                                        </span>
                                    </div>
                                </div>

                                <div class="exam-card-body">
                                    <?php if ($exam['description']): ?>
                                        <p class="exam-description">
                                            <?php echo htmlspecialchars(substr($exam['description'], 0, 120)); ?><?php echo strlen($exam['description']) > 120 ? '...' : ''; ?>
                                        </p>
                                    <?php endif; ?>

                                    <div class="exam-stats">
                                        <div class="stat-item">
                                            <div class="stat-value"><?php echo $exam['question_count']; ?></div>
                                            <div class="stat-label">Questions</div>
                                        </div>
                                        <div class="stat-item">
                                            <div class="stat-value"><?php echo $exam['total_points']; ?></div>
                                            <div class="stat-label">Points</div>
                                        </div>
                                        <div class="stat-item">
                                            <div class="stat-value"><?php echo $exam['attempt_count']; ?></div>
                                            <div class="stat-label">Attempts</div>
                                        </div>
                                    </div>

                                    <div class="exam-details">
                                        <?php if ($exam['time_limit']): ?>
                                            <div class="detail-item">
                                                <i class="fas fa-clock"></i>
                                                <span><?php echo $exam['time_limit']; ?> min</span>
                                            </div>
                                        <?php endif; ?>
                                        <?php if ($exam['passing_score']): ?>
                                            <div class="detail-item">
                                                <i class="fas fa-trophy"></i>
                                                <span><?php echo $exam['passing_score']; ?>% pass</span>
                                            </div>
                                        <?php endif; ?>
                                        <?php if ($exam['course_title']): ?>
                                            <div class="detail-item">
                                                <i class="fas fa-book"></i>
                                                <span><?php echo htmlspecialchars($exam['course_title']); ?></span>
                                            </div>
                                        <?php endif; ?>
                                    </div>
                                </div>

                                <div class="exam-card-footer">
                                    <div class="exam-actions">
                                        <button class="action-btn edit" onclick="window.location.href='edit.php?id=<?php echo $exam['id']; ?>'" title="Edit Exam">
                                            <i class="fas fa-edit"></i>
                                            <span>Edit</span>
                                        </button>
                                        <button class="action-btn preview" onclick="previewExam(<?php echo $exam['id']; ?>)" title="Preview Exam">
                                            <i class="fas fa-eye"></i>
                                            <span>Preview</span>
                                        </button>
                                        <button class="action-btn results" onclick="viewResults(<?php echo $exam['id']; ?>)" title="View Results">
                                            <i class="fas fa-chart-bar"></i>
                                            <span>Results</span>
                                        </button>
                                        <div class="dropdown">
                                            <button class="action-btn more" type="button" data-bs-toggle="dropdown" title="More Options">
                                                <i class="fas fa-ellipsis-v"></i>
                                            </button>
                                            <ul class="dropdown-menu">
                                                <li><a class="dropdown-item" href="#" onclick="duplicateExam(<?php echo $exam['id']; ?>)">
                                                    <i class="fas fa-copy me-2"></i>Duplicate
                                                </a></li>
                                                <li><a class="dropdown-item" href="#" onclick="exportExam(<?php echo $exam['id']; ?>)">
                                                    <i class="fas fa-download me-2"></i>Export
                                                </a></li>
                                                <li><hr class="dropdown-divider"></li>
                                                <li><a class="dropdown-item text-danger" href="#" onclick="deleteExam(<?php echo $exam['id']; ?>)">
                                                    <i class="fas fa-trash me-2"></i>Delete
                                                </a></li>
                                            </ul>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>
            </div>
    </div>
</div>

<script>
function previewExam(examId) {
    window.open('preview.php?id=' + examId, '_blank');
}

function viewResults(examId) {
    window.location.href = 'results.php?id=' + examId;
}

function duplicateExam(examId) {
    if (confirm('Are you sure you want to duplicate this exam?')) {
        // Implement duplicate functionality
        alert('Duplicate functionality to be implemented');
    }
}

function exportExam(examId) {
    // Implement export functionality
    alert('Export functionality to be implemented');
}

function deleteExam(examId) {
    if (confirm('Are you sure you want to delete this exam? This action cannot be undone.')) {
        // Implement delete functionality
        alert('Delete functionality to be implemented');
    }
}
</script>

<style>
/* Custom styles for Exams Index page */
.admin-container {
    display: flex;
    min-height: 100vh;
    background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
}

.admin-sidebar {
    width: 280px;
    background: linear-gradient(135deg, #1e293b 0%, #334155 100%);
    color: white;
    position: fixed;
    height: 100vh;
    overflow-y: auto;
    box-shadow: 4px 0 15px rgba(0, 0, 0, 0.1);
    z-index: 1000;
}

.sidebar-header {
    padding: 2rem 1.5rem;
    border-bottom: 1px solid #374151;
    text-align: center;
}

.sidebar-header h2 {
    font-size: 1.5rem;
    font-weight: 800;
    margin-bottom: 0.5rem;
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 0.5rem;
}

.sidebar-nav ul {
    list-style: none;
    padding: 0;
    margin: 0;
}

.sidebar-nav li a {
    display: flex;
    align-items: center;
    gap: 0.75rem;
    padding: 1rem 1.5rem;
    color: #cbd5e1;
    text-decoration: none;
    transition: all 0.3s ease;
    font-weight: 500;
    border-left: 3px solid transparent;
}

.sidebar-nav li a:hover,
.sidebar-nav li a.active {
    background: linear-gradient(135deg, #374151 0%, #4b5563 100%);
    color: white;
    border-left-color: #3b82f6;
    transform: translateX(5px);
}

.admin-main {
    margin-left: 280px;
    flex: 1;
    min-height: 100vh;
}

.admin-header {
    background: white;
    padding: 2rem;
    border-bottom: 1px solid #e5e7eb;
    box-shadow: 0 2px 10px rgba(0, 0, 0, 0.05);
}

.admin-header h1 {
    font-size: 2rem;
    font-weight: 800;
    color: #1f2937;
    margin-bottom: 0.5rem;
    display: flex;
    align-items: center;
    gap: 0.75rem;
}

.header-actions {
    display: flex;
    gap: 1rem;
    align-items: center;
}

.admin-content {
    padding: 2rem;
}

.filter-section {
    background: white;
    border-radius: 1rem;
    padding: 2rem;
    margin-bottom: 2rem;
    box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
    border: 1px solid rgba(255, 255, 255, 0.8);
}

.filter-row {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 1.5rem;
    align-items: end;
}

.filter-group label {
    font-weight: 600;
    color: #374151;
    margin-bottom: 0.5rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.filter-actions {
    display: flex;
    gap: 0.75rem;
    align-items: end;
}

.exams-grid-container {
    background: white;
    border-radius: 1rem;
    box-shadow: 0 8px 32px rgba(0, 0, 0, 0.1);
    overflow: hidden;
    border: 1px solid rgba(255, 255, 255, 0.8);
}

.grid-header {
    padding: 2rem;
    background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
    border-bottom: 1px solid #e5e7eb;
}

.grid-header h3 {
    font-size: 1.5rem;
    font-weight: 700;
    color: #1f2937;
    margin: 0;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.exams-grid {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(380px, 1fr));
    gap: 2rem;
    padding: 2rem;
}

.exam-card {
    background: white;
    border-radius: 1rem;
    box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
    border: 1px solid #e5e7eb;
    overflow: hidden;
    transition: all 0.3s ease;
}

.exam-card:hover {
    transform: translateY(-5px);
    box-shadow: 0 12px 25px rgba(0, 0, 0, 0.15);
}

.exam-card-header {
    padding: 1.5rem;
    background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
    border-bottom: 1px solid #e5e7eb;
    display: flex;
    justify-content: space-between;
    align-items: flex-start;
    gap: 1rem;
}

.exam-title-section h4 {
    font-size: 1.25rem;
    font-weight: 700;
    color: #1f2937;
    margin: 0 0 0.5rem 0;
    line-height: 1.3;
}

.exam-type-badge {
    display: inline-flex;
    align-items: center;
    padding: 0.25rem 0.75rem;
    border-radius: 1rem;
    font-size: 0.75rem;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.05em;
}

.exam-type-badge.quiz {
    background: linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%);
    color: white;
}

.exam-type-badge.midterm {
    background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%);
    color: white;
}

.exam-type-badge.final {
    background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%);
    color: white;
}

.exam-type-badge.practice {
    background: linear-gradient(135deg, #10b981 0%, #059669 100%);
    color: white;
}

.exam-type-badge.certification {
    background: linear-gradient(135deg, #8b5cf6 0%, #7c3aed 100%);
    color: white;
}

.status-badge {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.5rem 1rem;
    border-radius: 2rem;
    font-size: 0.875rem;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.05em;
}

.status-badge.active {
    background: linear-gradient(135deg, #10b981 0%, #059669 100%);
    color: white;
    box-shadow: 0 4px 15px rgba(16, 185, 129, 0.3);
}

.status-badge.draft {
    background: linear-gradient(135deg, #6b7280 0%, #4b5563 100%);
    color: white;
    box-shadow: 0 4px 15px rgba(107, 114, 128, 0.3);
}

.status-badge.archived {
    background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%);
    color: white;
    box-shadow: 0 4px 15px rgba(245, 158, 11, 0.3);
}

.exam-card-body {
    padding: 1.5rem;
}

.exam-description {
    color: #64748b;
    font-size: 0.9rem;
    line-height: 1.5;
    margin-bottom: 1.5rem;
}

.exam-stats {
    display: grid;
    grid-template-columns: repeat(3, 1fr);
    gap: 1rem;
    margin-bottom: 1.5rem;
}

.stat-item {
    text-align: center;
    padding: 1rem;
    background: #f8fafc;
    border-radius: 0.75rem;
    border: 1px solid #e5e7eb;
}

.stat-value {
    font-size: 1.5rem;
    font-weight: 800;
    color: #1f2937;
    margin-bottom: 0.25rem;
}

.stat-label {
    font-size: 0.875rem;
    color: #64748b;
    font-weight: 500;
    text-transform: uppercase;
    letter-spacing: 0.05em;
}

.exam-details {
    display: flex;
    flex-wrap: wrap;
    gap: 1rem;
    margin-bottom: 1.5rem;
}

.detail-item {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    font-size: 0.875rem;
    color: #64748b;
}

.detail-item i {
    color: #3b82f6;
    width: 16px;
}

.exam-card-footer {
    padding: 1.5rem;
    background: #f8fafc;
    border-top: 1px solid #e5e7eb;
}

.exam-actions {
    display: flex;
    gap: 0.5rem;
    justify-content: center;
}

.action-btn {
    padding: 0.75rem;
    border-radius: 0.5rem;
    font-size: 0.875rem;
    font-weight: 600;
    text-decoration: none;
    border: none;
    cursor: pointer;
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 0.25rem;
    transition: all 0.3s ease;
    min-width: 60px;
}

.action-btn.edit {
    background: linear-gradient(135deg, #3b82f6 0%, #2563eb 100%);
    color: white;
    box-shadow: 0 4px 15px rgba(59, 130, 246, 0.3);
}

.action-btn.edit:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(59, 130, 246, 0.4);
}

.action-btn.preview {
    background: linear-gradient(135deg, #6b7280 0%, #4b5563 100%);
    color: white;
    box-shadow: 0 4px 15px rgba(107, 114, 128, 0.3);
}

.action-btn.preview:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(107, 114, 128, 0.4);
}

.action-btn.results {
    background: linear-gradient(135deg, #10b981 0%, #059669 100%);
    color: white;
    box-shadow: 0 4px 15px rgba(16, 185, 129, 0.3);
}

.action-btn.results:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(16, 185, 129, 0.4);
}

.action-btn.more {
    background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
    color: #64748b;
    border: 1px solid #e5e7eb;
}

.action-btn.more:hover {
    background: #e2e8f0;
    transform: translateY(-2px);
}

.action-btn span {
    font-size: 0.75rem;
    font-weight: 500;
}

.empty-state {
    text-align: center;
    padding: 4rem 2rem;
    background: white;
    border-radius: 1rem;
    box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
}

.empty-state-icon {
    width: 80px;
    height: 80px;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    margin: 0 auto 2rem;
    color: white;
    font-size: 2rem;
    box-shadow: 0 8px 32px rgba(102, 126, 234, 0.3);
}

.empty-state h3 {
    font-size: 1.5rem;
    font-weight: 700;
    color: #1f2937;
    margin-bottom: 1rem;
}

.empty-state p {
    color: #64748b;
    font-size: 1.1rem;
    margin-bottom: 2rem;
}

.btn-primary {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    border: none;
    border-radius: 0.75rem;
    padding: 0.75rem 1.5rem;
    font-weight: 600;
    transition: all 0.3s ease;
    box-shadow: 0 4px 15px rgba(102, 126, 234, 0.3);
}

.btn-primary:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(102, 126, 234, 0.4);
    background: linear-gradient(135deg, #5a67d8 0%, #6b46c1 100%);
}

.btn-success {
    background: linear-gradient(135deg, #10b981 0%, #059669 100%);
    border: none;
    border-radius: 0.75rem;
    padding: 0.75rem 1.5rem;
    font-weight: 600;
    transition: all 0.3s ease;
    box-shadow: 0 4px 15px rgba(16, 185, 129, 0.3);
}

.btn-success:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(16, 185, 129, 0.4);
    background: linear-gradient(135deg, #059669 0%, #047857 100%);
}

.btn-outline-secondary {
    background: transparent;
    border: 2px solid #6b7280;
    color: #6b7280;
    border-radius: 0.75rem;
    padding: 0.75rem 1.5rem;
    font-weight: 600;
    transition: all 0.3s ease;
}

.btn-outline-secondary:hover {
    background: #6b7280;
    color: white;
    transform: translateY(-2px);
    box-shadow: 0 4px 15px rgba(107, 114, 128, 0.3);
}

.btn-lg {
    padding: 0.875rem 2rem;
    font-size: 1rem;
}

.form-select {
    width: 100%;
    padding: 0.75rem 1rem;
    border: 2px solid #e5e7eb;
    border-radius: 0.75rem;
    font-size: 0.875rem;
    transition: all 0.3s ease;
    background: white;
}

.form-select:focus {
    outline: none;
    border-color: #667eea;
    box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
}

.form-control {
    width: 100%;
    padding: 0.75rem 1rem;
    border: 2px solid #e5e7eb;
    border-radius: 0.75rem;
    font-size: 0.875rem;
    transition: all 0.3s ease;
    background: white;
}

.form-control:focus {
    outline: none;
    border-color: #667eea;
    box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
}

/* Responsive Design */
@media (max-width: 1024px) {
    .admin-sidebar {
        transform: translateX(-100%);
        transition: transform 0.3s ease;
    }

    .admin-main {
        margin-left: 0;
    }

    .filter-row {
        grid-template-columns: 1fr;
    }

    .exams-grid {
        grid-template-columns: 1fr;
        gap: 1.5rem;
    }

    .exam-card-header {
        flex-direction: column;
        align-items: stretch;
        gap: 1rem;
    }
}

@media (max-width: 768px) {
    .admin-header {
        padding: 1.5rem;
    }

    .admin-header h1 {
        font-size: 1.5rem;
    }

    .header-actions {
        flex-direction: column;
        align-items: stretch;
        gap: 0.5rem;
    }

    .admin-content {
        padding: 1rem;
    }

    .filter-section {
        padding: 1.5rem;
    }

    .exams-grid {
        padding: 1rem;
    }

    .exam-stats {
        grid-template-columns: 1fr;
    }

    .exam-actions {
        flex-wrap: wrap;
    }

    .action-btn {
        min-width: 50px;
        padding: 0.5rem;
    }

    .action-btn span {
        display: none;
    }

    .empty-state {
        padding: 3rem 1rem;
    }

    .empty-state-icon {
        width: 60px;
        height: 60px;
        font-size: 1.5rem;
    }
}
</style>