<?php
require_once '../../config/database.php';
require_once '../../config/auth.php';
require_once '../../config/rbac.php';
require_once '../../includes/functions.php';
require_once '../../includes/middleware.php';

// Check if user is instructor or admin
anyRoleMiddleware(['instructor', 'admin']);

$user = getUserContext();
$db = getDB();
$userId = $_SESSION['user_id'];

// Get courses and questions for dropdowns
$courses = $db->prepare("SELECT id, title FROM courses WHERE instructor_id = ? ORDER BY title", [$userId])->fetchAll();

// Get available questions (own questions + shared questions)
$questions = $db->prepare("
    SELECT q.id, q.title, q.question_type, q.difficulty, q.points, qc.name as category_name, c.title as course_title
    FROM questions q
    LEFT JOIN question_categories qc ON q.category_id = qc.id
    LEFT JOIN courses c ON q.course_id = c.id
    WHERE (q.instructor_id = ? OR q.is_shared = TRUE) AND q.status = 'approved'
    ORDER BY q.created_at DESC
", [$userId])->fetchAll();

$page_title = 'Create Exam - Mutalex Academy';
include '../../includes/header.php';
?>

<div class="admin-container">
    <div class="admin-sidebar">
        <div class="sidebar-header">
            <h2><i class="fas fa-graduation-cap me-2"></i>Mutalex Academy</h2>
            <p>Instructor Panel</p>
        </div>

        <nav class="sidebar-nav">
            <ul>
                <li><a href="../dashboard.php"><i class="fas fa-tachometer-alt me-2"></i>Dashboard</a></li>
                <li><a href="../courses/"><i class="fas fa-book me-2"></i>My Courses</a></li>
                <li><a href="../courses/new.php"><i class="fas fa-plus me-2"></i>Create Course</a></li>
                <li><a href="../questions/index.php"><i class="fas fa-question-circle me-2"></i>Question Bank</a></li>
                <li><a href="index.php" class="active"><i class="fas fa-file-alt me-2"></i>Exams</a></li>
                <li><a href="../resources.php"><i class="fas fa-folder me-2"></i>Resources</a></li>
                <li><a href="../students.php"><i class="fas fa-users me-2"></i>Students</a></li>
                <li><a href="../assignments.php"><i class="fas fa-tasks me-2"></i>Assignments</a></li>
                <li><a href="/lms/instructor/grading/index.php"><i class="fas fa-chart-line me-2"></i>Grades</a></li>
                <li><a href="../../logout.php"><i class="fas fa-sign-out-alt me-2"></i>Logout</a></li>
            </ul>
        </nav>
    </div>

    <div class="admin-main">
        <div class="admin-header">
            <div class="d-flex justify-content-between align-items-center">
                <div>
                    <h1><i class="fas fa-plus-circle text-success me-3"></i>Create Exam</h1>
                    <p class="text-muted mb-0">Design comprehensive assessments to evaluate student learning</p>
                </div>
                <a href="index.php" class="btn btn-outline-secondary btn-lg">
                    <i class="fas fa-arrow-left me-2"></i>Back to Exams
                </a>
            </div>
        </div>

        <div class="admin-content">

            <form id="examForm" method="POST" action="save.php">
                <div class="exam-form-layout">
                    <!-- Main Exam Form -->
                    <div class="exam-main-form">
                        <!-- Basic Information -->
                        <div class="form-card">
                            <div class="form-card-header">
                                <h3><i class="fas fa-info-circle me-2"></i>Exam Information</h3>
                                <p>Define the basic details and settings for your exam</p>
                            </div>
                            <div class="form-card-body">
                            <div class="row">
                                <div class="col-md-8">
                                    <div class="mb-3">
                                        <label for="title" class="form-label">Exam Title <span class="text-danger">*</span></label>
                                        <input type="text" class="form-control" id="title" name="title" required
                                               placeholder="Enter exam title">
                                    </div>
                                </div>
                                <div class="col-md-4">
                                    <div class="mb-3">
                                        <label for="exam_type" class="form-label">Exam Type</label>
                                        <select class="form-select" id="exam_type" name="exam_type">
                                            <option value="quiz">Quiz</option>
                                            <option value="midterm">Midterm</option>
                                            <option value="final">Final</option>
                                            <option value="practice">Practice</option>
                                            <option value="certification">Certification</option>
                                        </select>
                                    </div>
                                </div>
                            </div>

                            <div class="mb-3">
                                <label for="description" class="form-label">Description</label>
                                <textarea class="form-control" id="description" name="description" rows="3"
                                          placeholder="Describe the exam purpose and instructions"></textarea>
                            </div>

                            <div class="mb-3">
                                <label for="instructions" class="form-label">Instructions for Students</label>
                                <textarea class="form-control" id="instructions" name="instructions" rows="4"
                                          placeholder="Special instructions, rules, or guidelines for students"></textarea>
                            </div>
                        </div>
                    </div>

                        <!-- Question Selection -->
                        <div class="form-card">
                            <div class="form-card-header d-flex justify-content-between align-items-center">
                                <div>
                                    <h3><i class="fas fa-question-circle me-2"></i>Question Selection</h3>
                                    <p>Choose questions from your question bank to include in this exam</p>
                                </div>
                                <button type="button" class="btn btn-primary" onclick="openQuestionBank()">
                                    <i class="fas fa-plus me-2"></i>Add Questions
                                </button>
                            </div>
                            <div class="form-card-body">
                            <div id="selected_questions" class="mb-3">
                                <p class="text-muted">No questions selected yet. Click "Add Questions" to choose from your question bank.</p>
                            </div>

                            <!-- Question sorting and organization -->
                            <div class="mb-3">
                                <label class="form-label">Question Organization</label>
                                <div class="row">
                                    <div class="col-md-6">
                                        <div class="form-check">
                                            <input class="form-check-input" type="checkbox" id="shuffle_questions" name="shuffle_questions">
                                            <label class="form-check-label" for="shuffle_questions">
                                                Shuffle question order
                                            </label>
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="form-check">
                                            <input class="form-check-input" type="checkbox" id="shuffle_options" name="shuffle_options">
                                            <label class="form-check-label" for="shuffle_options">
                                                Shuffle answer options
                                            </label>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                    <!-- Settings Sidebar -->
                    <div class="exam-settings-sidebar">
                        <!-- Timing and Scoring -->
                        <div class="form-card">
                            <div class="form-card-header">
                                <h3><i class="fas fa-clock me-2"></i>Timing & Scoring</h3>
                                <p>Configure time limits and grading criteria</p>
                            </div>
                            <div class="form-card-body">
                            <div class="mb-3">
                                <label for="time_limit" class="form-label">Time Limit (minutes)</label>
                                <input type="number" class="form-control" id="time_limit" name="time_limit" min="0"
                                       placeholder="Leave empty for no limit">
                                <div class="form-text">Total time allowed for the exam</div>
                            </div>

                            <div class="mb-3">
                                <label for="passing_score" class="form-label">Passing Score (%)</label>
                                <input type="number" class="form-control" id="passing_score" name="passing_score"
                                       min="0" max="100" value="50">
                            </div>

                            <div class="mb-3">
                                <label for="max_attempts" class="form-label">Maximum Attempts</label>
                                <input type="number" class="form-control" id="max_attempts" name="max_attempts"
                                       min="1" value="1">
                            </div>

                            <div class="mb-3">
                                <label for="total_points" class="form-label">Total Points</label>
                                <input type="number" class="form-control" id="total_points" name="total_points"
                                       readonly value="0">
                                <div class="form-text">Calculated from selected questions</div>
                            </div>
                        </div>
                    </div>

                        <!-- Access Control -->
                        <div class="form-card">
                            <div class="form-card-header">
                                <h3><i class="fas fa-lock me-2"></i>Access Control</h3>
                                <p>Control when and how students can access this exam</p>
                            </div>
                            <div class="form-card-body">
                            <div class="mb-3">
                                <label for="course_id" class="form-label">Associated Course</label>
                                <select class="form-select" id="course_id" name="course_id">
                                    <option value="">General exam (not course-specific)</option>
                                    <?php foreach ($courses as $course): ?>
                                        <option value="<?php echo $course['id']; ?>">
                                            <?php echo htmlspecialchars($course['title']); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>

                            <div class="mb-3">
                                <label class="form-label">Schedule</label>
                                <div class="row">
                                    <div class="col-6">
                                        <input type="datetime-local" class="form-control" id="start_date" name="start_date"
                                               placeholder="Start date">
                                    </div>
                                    <div class="col-6">
                                        <input type="datetime-local" class="form-control" id="end_date" name="end_date"
                                               placeholder="End date">
                                    </div>
                                </div>
                                <div class="form-text">Leave empty for immediate availability</div>
                            </div>
                        </div>
                    </div>

                        <!-- Security & Behavior -->
                        <div class="form-card">
                            <div class="form-card-header">
                                <h3><i class="fas fa-shield-alt me-2"></i>Security & Behavior</h3>
                                <p>Configure exam security and student experience settings</p>
                            </div>
                            <div class="form-card-body">
                            <div class="mb-3">
                                <div class="form-check">
                                    <input class="form-check-input" type="checkbox" id="is_proctored" name="is_proctored">
                                    <label class="form-check-label" for="is_proctored">
                                        Enable proctoring
                                    </label>
                                </div>
                            </div>

                            <div class="mb-3">
                                <div class="form-check">
                                    <input class="form-check-input" type="checkbox" id="show_results" name="show_results" checked>
                                    <label class="form-check-label" for="show_results">
                                        Show results after completion
                                    </label>
                                </div>
                            </div>

                            <div class="mb-3">
                                <div class="form-check">
                                    <input class="form-check-input" type="checkbox" id="show_answers" name="show_answers">
                                    <label class="form-check-label" for="show_answers">
                                        Show correct answers
                                    </label>
                                </div>
                            </div>

                            <div class="mb-3">
                                <div class="form-check">
                                    <input class="form-check-input" type="checkbox" id="allow_review" name="allow_review" checked>
                                    <label class="form-check-label" for="allow_review">
                                        Allow review of answers
                                    </label>
                                </div>
                            </div>
                        </div>
                    </div>

                        <!-- Action Buttons -->
                        <div class="form-actions-card">
                            <div class="form-actions">
                                <button type="submit" name="action" value="save_draft" class="btn btn-outline-secondary">
                                    <i class="fas fa-save me-2"></i>Save as Draft
                                </button>
                                <button type="submit" name="action" value="publish" class="btn btn-primary">
                                    <i class="fas fa-globe me-2"></i>Publish Exam
                                </button>
                                <button type="button" class="btn btn-outline-info" onclick="previewExam()">
                                    <i class="fas fa-eye me-2"></i>Preview Exam
                                </button>
                            </div>
                        </div>
                </div>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Question Bank Modal -->
<div class="modal fade" id="questionBankModal" tabindex="-1">
    <div class="modal-dialog modal-xl">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Select Questions</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <!-- Question filters -->
                <div class="mb-3">
                    <div class="row">
                        <div class="col-md-3">
                            <select class="form-select" id="qb_type_filter">
                                <option value="">All Types</option>
                                <option value="mcq_single">MCQ Single</option>
                                <option value="mcq_multiple">MCQ Multiple</option>
                                <option value="true_false">True/False</option>
                                <option value="essay">Essay</option>
                                <option value="short_answer">Short Answer</option>
                                <option value="matching">Matching</option>
                                <option value="code">Code</option>
                                <option value="fill_blanks">Fill Blanks</option>
                            </select>
                        </div>
                        <div class="col-md-3">
                            <select class="form-select" id="qb_difficulty_filter">
                                <option value="">All Difficulties</option>
                                <option value="easy">Easy</option>
                                <option value="medium">Medium</option>
                                <option value="hard">Hard</option>
                            </select>
                        </div>
                        <div class="col-md-6">
                            <input type="text" class="form-control" id="qb_search" placeholder="Search questions...">
                        </div>
                    </div>
                </div>

                <!-- Questions list -->
                <div class="question-bank-list" style="max-height: 400px; overflow-y: auto;">
                    <?php foreach ($questions as $question): ?>
                        <div class="question-item border rounded p-3 mb-2" data-type="<?php echo $question['question_type']; ?>" data-difficulty="<?php echo $question['difficulty']; ?>">
                            <div class="d-flex justify-content-between align-items-start">
                                <div class="flex-grow-1">
                                    <h6 class="mb-1"><?php echo htmlspecialchars($question['title']); ?></h6>
                                    <div class="mb-2">
                                        <span class="badge bg-secondary me-1"><?php echo ucfirst(str_replace('_', ' ', $question['question_type'])); ?></span>
                                        <span class="badge bg-<?php echo $question['difficulty'] === 'easy' ? 'success' : ($question['difficulty'] === 'medium' ? 'warning' : 'danger'); ?>">
                                            <?php echo ucfirst($question['difficulty']); ?>
                                        </span>
                                        <small class="text-muted ms-2"><?php echo $question['points']; ?> pts</small>
                                    </div>
                                    <?php if ($question['category_name']): ?>
                                        <small class="text-muted">Category: <?php echo htmlspecialchars($question['category_name']); ?></small>
                                    <?php endif; ?>
                                </div>
                                <button type="button" class="btn btn-outline-primary btn-sm add-question-btn"
                                        data-id="<?php echo $question['id']; ?>"
                                        data-title="<?php echo htmlspecialchars($question['title']); ?>"
                                        data-points="<?php echo $question['points']; ?>">
                                    <i class="fas fa-plus"></i> Add
                                </button>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
            </div>
        </div>
    </div>
</div>

<script>
// Selected questions storage
let selectedQuestions = [];
let totalPoints = 0;

// Question Bank Modal
function openQuestionBank() {
    const modal = new bootstrap.Modal(document.getElementById('questionBankModal'));
    modal.show();
}

// Add question to exam
document.addEventListener('click', function(e) {
    if (e.target.classList.contains('add-question-btn') || e.target.closest('.add-question-btn')) {
        const btn = e.target.classList.contains('add-question-btn') ? e.target : e.target.closest('.add-question-btn');
        const questionId = btn.dataset.id;
        const title = btn.dataset.title;
        const points = parseFloat(btn.dataset.points);

        // Check if already selected
        if (selectedQuestions.some(q => q.id == questionId)) {
            alert('Question already added to exam');
            return;
        }

        // Add to selected questions
        selectedQuestions.push({
            id: questionId,
            title: title,
            points: points,
            order: selectedQuestions.length + 1
        });

        totalPoints += points;
        updateSelectedQuestions();
        updateTotalPoints();

        // Hide button
        btn.style.display = 'none';
    }
});

// Update selected questions display
function updateSelectedQuestions() {
    const container = document.getElementById('selected_questions');

    if (selectedQuestions.length === 0) {
        container.innerHTML = '<p class="text-muted">No questions selected yet. Click "Add Questions" to choose from your question bank.</p>';
        return;
    }

    let html = '<div class="selected-questions-list">';
    html += '<h6>Selected Questions (' + selectedQuestions.length + ')</h6>';

    selectedQuestions.forEach((question, index) => {
        html += `
            <div class="question-order-item border rounded p-2 mb-2 d-flex justify-content-between align-items-center">
                <div class="flex-grow-1">
                    <span class="fw-bold">${index + 1}.</span> ${question.title}
                    <small class="text-muted ms-2">(${question.points} pts)</small>
                    <input type="hidden" name="question_ids[]" value="${question.id}">
                    <input type="hidden" name="question_orders[]" value="${index + 1}">
                    <input type="hidden" name="question_points[]" value="${question.points}">
                </div>
                <button type="button" class="btn btn-outline-danger btn-sm" onclick="removeQuestion(${index})">
                    <i class="fas fa-trash"></i>
                </button>
            </div>
        `;
    });

    html += '</div>';
    container.innerHTML = html;
}

// Remove question from exam
function removeQuestion(index) {
    const removedQuestion = selectedQuestions.splice(index, 1)[0];
    totalPoints -= removedQuestion.points;

    // Reorder remaining questions
    selectedQuestions.forEach((q, i) => {
        q.order = i + 1;
    });

    updateSelectedQuestions();
    updateTotalPoints();

    // Show add button again for removed question
    const addBtn = document.querySelector(`[data-id="${removedQuestion.id}"].add-question-btn`);
    if (addBtn) {
        addBtn.style.display = 'inline-block';
    }
}

// Update total points display
function updateTotalPoints() {
    document.getElementById('total_points').value = totalPoints;
}

// Question bank filtering
document.getElementById('qb_type_filter').addEventListener('change', filterQuestions);
document.getElementById('qb_difficulty_filter').addEventListener('change', filterQuestions);
document.getElementById('qb_search').addEventListener('input', filterQuestions);

function filterQuestions() {
    const typeFilter = document.getElementById('qb_type_filter').value;
    const difficultyFilter = document.getElementById('qb_difficulty_filter').value;
    const searchTerm = document.getElementById('qb_search').value.toLowerCase();

    const questions = document.querySelectorAll('.question-item');

    questions.forEach(question => {
        const type = question.dataset.type;
        const difficulty = question.dataset.difficulty;
        const title = question.textContent.toLowerCase();

        const typeMatch = !typeFilter || type === typeFilter;
        const difficultyMatch = !difficultyFilter || difficulty === difficultyFilter;
        const searchMatch = !searchTerm || title.includes(searchTerm);

        question.style.display = (typeMatch && difficultyMatch && searchMatch) ? 'block' : 'none';
    });
}

function previewExam() {
    if (selectedQuestions.length === 0) {
        alert('Please add at least one question to preview the exam');
        return;
    }

    // Basic preview - could be enhanced
    const previewData = {
        title: document.getElementById('title').value,
        questions: selectedQuestions,
        totalPoints: totalPoints
    };

    console.log('Exam Preview:', previewData);
    alert('Preview functionality - check console for exam data');
}
</script>

<style>
/* Custom styles for Create Exam page */
.admin-container {
    display: flex;
    min-height: 100vh;
    background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
}

.admin-sidebar {
    width: 280px;
    background: linear-gradient(135deg, #1e293b 0%, #334155 100%);
    color: white;
    position: fixed;
    height: 100vh;
    overflow-y: auto;
    box-shadow: 4px 0 15px rgba(0, 0, 0, 0.1);
    z-index: 1000;
}

.sidebar-header {
    padding: 2rem 1.5rem;
    border-bottom: 1px solid #374151;
    text-align: center;
}

.sidebar-header h2 {
    font-size: 1.5rem;
    font-weight: 800;
    margin-bottom: 0.5rem;
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 0.5rem;
}

.sidebar-nav ul {
    list-style: none;
    padding: 0;
    margin: 0;
}

.sidebar-nav li a {
    display: flex;
    align-items: center;
    gap: 0.75rem;
    padding: 1rem 1.5rem;
    color: #cbd5e1;
    text-decoration: none;
    transition: all 0.3s ease;
    font-weight: 500;
    border-left: 3px solid transparent;
}

.sidebar-nav li a:hover,
.sidebar-nav li a.active {
    background: linear-gradient(135deg, #374151 0%, #4b5563 100%);
    color: white;
    border-left-color: #3b82f6;
    transform: translateX(5px);
}

.admin-main {
    margin-left: 280px;
    flex: 1;
    min-height: 100vh;
}

.admin-header {
    background: white;
    padding: 2rem;
    border-bottom: 1px solid #e5e7eb;
    box-shadow: 0 2px 10px rgba(0, 0, 0, 0.05);
}

.admin-header h1 {
    font-size: 2rem;
    font-weight: 800;
    color: #1f2937;
    margin-bottom: 0.5rem;
    display: flex;
    align-items: center;
    gap: 0.75rem;
}

.admin-content {
    padding: 2rem;
}

.exam-form-layout {
    display: grid;
    grid-template-columns: 2fr 1fr;
    gap: 2rem;
    align-items: start;
}

.exam-main-form {
    display: flex;
    flex-direction: column;
    gap: 2rem;
}

.exam-settings-sidebar {
    position: sticky;
    top: 2rem;
    display: flex;
    flex-direction: column;
    gap: 2rem;
}

.form-card {
    background: white;
    border-radius: 1rem;
    box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
    border: 1px solid rgba(255, 255, 255, 0.8);
    overflow: hidden;
}

.form-card-header {
    padding: 2rem;
    background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
    border-bottom: 1px solid #e5e7eb;
}

.form-card-header h3 {
    font-size: 1.25rem;
    font-weight: 700;
    color: #1f2937;
    margin-bottom: 0.5rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.form-card-header p {
    color: #64748b;
    margin: 0;
    font-size: 0.9rem;
}

.form-card-header.d-flex {
    display: flex !important;
    justify-content: space-between;
    align-items: center;
}

.form-card-body {
    padding: 2rem;
}

.form-group {
    margin-bottom: 1.5rem;
}

.form-group label {
    display: block;
    font-weight: 600;
    color: #374151;
    margin-bottom: 0.5rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.form-control {
    width: 100%;
    padding: 0.75rem 1rem;
    border: 2px solid #e5e7eb;
    border-radius: 0.75rem;
    font-size: 0.875rem;
    transition: all 0.3s ease;
    background: white;
}

.form-control:focus {
    outline: none;
    border-color: #667eea;
    box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
}

.form-select {
    width: 100%;
    padding: 0.75rem 1rem;
    border: 2px solid #e5e7eb;
    border-radius: 0.75rem;
    font-size: 0.875rem;
    transition: all 0.3s ease;
    background: white;
}

.form-select:focus {
    outline: none;
    border-color: #667eea;
    box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
}

.form-text {
    color: #64748b;
    font-size: 0.875rem;
    margin-top: 0.25rem;
}

.form-check {
    display: flex;
    align-items: center;
    gap: 0.75rem;
    margin-bottom: 1rem;
}

.form-check-input {
    width: 1.25rem;
    height: 1.25rem;
    border: 2px solid #e5e7eb;
    border-radius: 0.375rem;
    background: white;
    cursor: pointer;
    transition: all 0.3s ease;
}

.form-check-input:checked {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    border-color: #667eea;
}

.form-check-label {
    color: #374151;
    font-weight: 500;
    cursor: pointer;
}

.btn {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.75rem 1.5rem;
    font-weight: 600;
    text-decoration: none;
    border-radius: 0.75rem;
    border: none;
    cursor: pointer;
    transition: all 0.3s ease;
    font-size: 0.875rem;
}

.btn-primary {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    box-shadow: 0 4px 15px rgba(102, 126, 234, 0.3);
}

.btn-primary:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(102, 126, 234, 0.4);
    background: linear-gradient(135deg, #5a67d8 0%, #6b46c1 100%);
}

.btn-outline-primary {
    background: transparent;
    border: 2px solid #667eea;
    color: #667eea;
}

.btn-outline-primary:hover {
    background: #667eea;
    color: white;
    transform: translateY(-2px);
    box-shadow: 0 4px 15px rgba(102, 126, 234, 0.3);
}

.btn-outline-secondary {
    background: transparent;
    border: 2px solid #6b7280;
    color: #6b7280;
}

.btn-outline-secondary:hover {
    background: #6b7280;
    color: white;
    transform: translateY(-2px);
    box-shadow: 0 4px 15px rgba(107, 114, 128, 0.3);
}

.btn-outline-info {
    background: transparent;
    border: 2px solid #3b82f6;
    color: #3b82f6;
}

.btn-outline-info:hover {
    background: #3b82f6;
    color: white;
    transform: translateY(-2px);
    box-shadow: 0 4px 15px rgba(59, 130, 246, 0.3);
}

.btn-lg {
    padding: 0.875rem 2rem;
    font-size: 1rem;
}

.text-danger {
    color: #dc2626 !important;
}

.text-muted {
    color: #64748b !important;
}

.form-actions-card {
    background: white;
    border-radius: 1rem;
    box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
    border: 1px solid rgba(255, 255, 255, 0.8);
    overflow: hidden;
}

.form-actions {
    padding: 2rem;
    display: flex;
    gap: 1rem;
    justify-content: center;
    flex-wrap: wrap;
}

/* Question Bank Modal Enhancements */
.modal-content {
    border-radius: 1rem;
    border: none;
    box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
}

.modal-header {
    background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
    border-radius: 1rem 1rem 0 0;
    border-bottom: 1px solid #e5e7eb;
    padding: 1.5rem 2rem;
}

.modal-title {
    font-weight: 700;
    color: #1f2937;
    font-size: 1.25rem;
}

.modal-body {
    padding: 2rem;
}

.question-bank-list {
    max-height: 400px;
    overflow-y: auto;
    border: 1px solid #e5e7eb;
    border-radius: 0.75rem;
    background: #f8fafc;
}

.question-item {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 1rem 1.5rem;
    border-bottom: 1px solid #e5e7eb;
    transition: all 0.3s ease;
}

.question-item:hover {
    background: white;
    transform: translateX(5px);
}

.question-item:last-child {
    border-bottom: none;
}

.question-item h6 {
    margin: 0 0 0.5rem 0;
    font-size: 1rem;
    font-weight: 600;
    color: #1f2937;
}

.question-item .badge {
    font-size: 0.75rem;
    margin-right: 0.5rem;
}

.add-question-btn {
    padding: 0.5rem 1rem;
    border-radius: 0.5rem;
    font-size: 0.875rem;
    font-weight: 600;
    background: linear-gradient(135deg, #10b981 0%, #059669 100%);
    color: white;
    border: none;
    cursor: pointer;
    transition: all 0.3s ease;
}

.add-question-btn:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 15px rgba(16, 185, 129, 0.3);
}

/* Selected Questions Styling */
.selected-questions-list {
    margin-top: 1rem;
}

.question-order-item {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 1rem;
    background: #f8fafc;
    border: 1px solid #e5e7eb;
    border-radius: 0.75rem;
    margin-bottom: 0.5rem;
}

.question-order-item strong {
    color: #667eea;
    margin-right: 0.5rem;
}

/* Responsive Design */
@media (max-width: 1024px) {
    .admin-sidebar {
        transform: translateX(-100%);
        transition: transform 0.3s ease;
    }

    .admin-main {
        margin-left: 0;
    }

    .exam-form-layout {
        grid-template-columns: 1fr;
        gap: 1.5rem;
    }

    .exam-settings-sidebar {
        position: static;
    }

    .form-card-body {
        padding: 1.5rem;
    }

    .form-actions {
        flex-direction: column;
    }

    .btn {
        justify-content: center;
    }
}

@media (max-width: 768px) {
    .admin-header {
        padding: 1.5rem;
    }

    .admin-header h1 {
        font-size: 1.5rem;
    }

    .admin-content {
        padding: 1rem;
    }

    .form-card-header {
        padding: 1.5rem;
    }

    .form-card-body {
        padding: 1.5rem;
    }

    .form-actions {
        padding: 1.5rem;
    }

    .modal-body {
        padding: 1.5rem;
    }

    .question-bank-list {
        max-height: 300px;
    }
}
</style>

<?php include '../../includes/footer.php'; ?>