<?php
require_once '../../config/database.php';
require_once '../../config/auth.php';
require_once '../../config/rbac.php';
require_once '../../includes/functions.php';
require_once '../../includes/middleware.php';

// Check if user is instructor or admin
anyRoleMiddleware(['instructor', 'admin']);

$user = getUserContext();
$db = getDB();
$userId = $_SESSION['user_id'];

// Handle filters and search
$search = $_GET['search'] ?? '';
$type_filter = $_GET['type'] ?? '';
$difficulty_filter = $_GET['difficulty'] ?? '';
$category_filter = $_GET['category'] ?? '';
$status_filter = $_GET['status'] ?? '';
$course_filter = $_GET['course'] ?? '';

// Build query
$query = "
    SELECT q.*, qc.name as category_name, c.title as course_title,
           GROUP_CONCAT(DISTINCT qt.name) as tag_names
    FROM questions q
    LEFT JOIN question_categories qc ON q.category_id = qc.id
    LEFT JOIN courses c ON q.course_id = c.id
    LEFT JOIN question_tag_assignments qta ON q.id = qta.question_id
    LEFT JOIN question_tags qt ON qta.tag_id = qt.id
    WHERE q.instructor_id = ? OR q.is_shared = TRUE
";

$params = [$userId];

if (!empty($search)) {
    $query .= " AND (q.title LIKE ? OR q.content LIKE ?)";
    $params[] = "%$search%";
    $params[] = "%$search%";
}

if (!empty($type_filter)) {
    $query .= " AND q.question_type = ?";
    $params[] = $type_filter;
}

if (!empty($difficulty_filter)) {
    $query .= " AND q.difficulty = ?";
    $params[] = $difficulty_filter;
}

if (!empty($category_filter)) {
    $query .= " AND q.category_id = ?";
    $params[] = $category_filter;
}

if (!empty($status_filter)) {
    $query .= " AND q.status = ?";
    $params[] = $status_filter;
}

if (!empty($course_filter)) {
    $query .= " AND q.course_id = ?";
    $params[] = $course_filter;
}

$query .= " GROUP BY q.id ORDER BY q.created_at DESC";

// Get questions
$stmt = $db->prepare($query);
$stmt->execute($params);
$questions = $stmt->fetchAll();

// Get filter options
$categories = $db->query("SELECT id, name FROM question_categories WHERE is_active = TRUE ORDER BY name")->fetchAll();
$courses = $db->prepare("SELECT id, title FROM courses WHERE instructor_id = ? ORDER BY title", [$userId])->fetchAll();

$page_title = 'Question Bank - Mutalex Academy';
?>

<div class="admin-container">
    <div class="admin-sidebar">
        <div class="sidebar-header">
            <h2><i class="fas fa-graduation-cap me-2"></i>Mutalex Academy</h2>
            <p>Instructor Panel</p>
        </div>

        <nav class="sidebar-nav">
            <ul>
                <li><a href="../dashboard.php"><i class="fas fa-tachometer-alt me-2"></i>Dashboard</a></li>
                <li><a href="../courses/"><i class="fas fa-book me-2"></i>My Courses</a></li>
                <li><a href="../courses/new.php"><i class="fas fa-plus me-2"></i>Create Course</a></li>
                <li><a href="index.php" class="active"><i class="fas fa-question-circle me-2"></i>Question Bank</a></li>
                <li><a href="../exams/"><i class="fas fa-file-alt me-2"></i>Exams</a></li>
                <li><a href="../resources.php"><i class="fas fa-folder me-2"></i>Resources</a></li>
                <li><a href="../students.php"><i class="fas fa-users me-2"></i>Students</a></li>
                <li><a href="../assignments.php"><i class="fas fa-tasks me-2"></i>Assignments</a></li>
                <li><a href="/lms/instructor/grading/index.php"><i class="fas fa-chart-line me-2"></i>Grades</a></li>
                <li><a href="../../logout.php"><i class="fas fa-sign-out-alt me-2"></i>Logout</a></li>
            </ul>
        </nav>
    </div>

    <div class="admin-main">
        <div class="admin-header">
            <div class="d-flex justify-content-between align-items-center">
                <div>
                    <h1><i class="fas fa-question-circle text-primary me-3"></i>Question Bank</h1>
                    <p class="text-muted mb-0">Manage your assessment questions with advanced analytics</p>
                </div>
                <div class="header-actions">
                    <a href="new.php" class="btn btn-primary btn-lg">
                        <i class="fas fa-plus me-2"></i>Create Question
                    </a>
                    <a href="import.php" class="btn btn-success btn-lg">
                        <i class="fas fa-upload me-2"></i>Import
                    </a>
                    <button class="btn btn-outline-primary btn-lg" onclick="exportQuestions()">
                        <i class="fas fa-download me-2"></i>Export
                    </button>
                </div>
            </div>
        </div>

        <div class="admin-content">

            <!-- Filters -->
            <div class="filter-section">
                <div class="filter-form">
                    <div class="filter-row">
                        <div class="filter-group">
                            <label for="search"><i class="fas fa-search me-1"></i>Search</label>
                            <input type="text" id="search" class="form-control" name="search" placeholder="Search questions..."
                                   value="<?php echo htmlspecialchars($search); ?>">
                        </div>
                        <div class="filter-group">
                            <label for="type"><i class="fas fa-list me-1"></i>Question Type</label>
                            <select class="form-select" id="type" name="type">
                                <option value="">All Types</option>
                                <option value="mcq_single" <?php echo $type_filter === 'mcq_single' ? 'selected' : ''; ?>>Multiple Choice (Single)</option>
                                <option value="mcq_multiple" <?php echo $type_filter === 'mcq_multiple' ? 'selected' : ''; ?>>Multiple Choice (Multiple)</option>
                                <option value="true_false" <?php echo $type_filter === 'true_false' ? 'selected' : ''; ?>>True/False</option>
                                <option value="essay" <?php echo $type_filter === 'essay' ? 'selected' : ''; ?>>Essay</option>
                                <option value="short_answer" <?php echo $type_filter === 'short_answer' ? 'selected' : ''; ?>>Short Answer</option>
                                <option value="matching" <?php echo $type_filter === 'matching' ? 'selected' : ''; ?>>Matching</option>
                                <option value="code" <?php echo $type_filter === 'code' ? 'selected' : ''; ?>>Code</option>
                                <option value="fill_blanks" <?php echo $type_filter === 'fill_blanks' ? 'selected' : ''; ?>>Fill in Blanks</option>
                            </select>
                        </div>
                        <div class="filter-group">
                            <label for="difficulty"><i class="fas fa-chart-line me-1"></i>Difficulty</label>
                            <select class="form-select" id="difficulty" name="difficulty">
                                <option value="">All Levels</option>
                                <option value="easy" <?php echo $difficulty_filter === 'easy' ? 'selected' : ''; ?>>Easy</option>
                                <option value="medium" <?php echo $difficulty_filter === 'medium' ? 'selected' : ''; ?>>Medium</option>
                                <option value="hard" <?php echo $difficulty_filter === 'hard' ? 'selected' : ''; ?>>Hard</option>
                            </select>
                        </div>
                        <div class="filter-group">
                            <label for="category"><i class="fas fa-tag me-1"></i>Category</label>
                            <select class="form-select" id="category" name="category">
                                <option value="">All Categories</option>
                                <?php foreach ($categories as $category): ?>
                                    <option value="<?php echo $category['id']; ?>" <?php echo $category_filter == $category['id'] ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($category['name']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="filter-group">
                            <label for="course"><i class="fas fa-book me-1"></i>Course</label>
                            <select class="form-select" id="course" name="course">
                                <option value="">All Courses</option>
                                <?php foreach ($courses as $course): ?>
                                    <option value="<?php echo $course['id']; ?>" <?php echo $course_filter == $course['id'] ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($course['title']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="filter-actions">
                            <button type="submit" class="btn btn-primary">
                                <i class="fas fa-filter me-1"></i>Apply Filters
                            </button>
                            <a href="index.php" class="btn btn-outline-secondary">
                                <i class="fas fa-times me-1"></i>Clear
                            </a>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Questions List -->
            <div class="questions-table-container">
                <div class="table-header">
                    <h3><i class="fas fa-list me-2"></i>Questions (<?php echo count($questions); ?>)</h3>
                    <div class="table-actions">
                        <a href="analysis.php" class="btn btn-outline-info">
                            <i class="fas fa-chart-bar me-1"></i>View Analysis
                        </a>
                    </div>
                </div>

                <?php if (empty($questions)): ?>
                    <div class="empty-state">
                        <div class="empty-state-icon">
                            <i class="fas fa-question-circle"></i>
                        </div>
                        <h3>No Questions Found</h3>
                        <p>Start building your question bank by creating your first assessment question.</p>
                        <a href="new.php" class="btn btn-primary btn-lg">
                            <i class="fas fa-plus me-2"></i>Create Your First Question
                        </a>
                    </div>
                <?php else: ?>
                    <div class="table-wrapper">
                        <table class="aligned-table">
                            <thead>
                                <tr>
                                    <th><i class="fas fa-file-alt me-1"></i>Question</th>
                                    <th><i class="fas fa-cogs me-1"></i>Type</th>
                                    <th><i class="fas fa-chart-line me-1"></i>Difficulty</th>
                                    <th><i class="fas fa-tag me-1"></i>Category</th>
                                    <th><i class="fas fa-info-circle me-1"></i>Status</th>
                                    <th><i class="fas fa-chart-bar me-1"></i>Usage</th>
                                    <th><i class="fas fa-tools me-1"></i>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($questions as $question): ?>
                                    <tr>
                                        <td>
                                            <div class="question-info">
                                                <div class="question-title"><?php echo htmlspecialchars(substr($question['title'], 0, 60)); ?><?php echo strlen($question['title']) > 60 ? '...' : ''; ?></div>
                                                <?php if ($question['course_title']): ?>
                                                    <div class="question-meta">
                                                        <i class="fas fa-book me-1"></i><?php echo htmlspecialchars($question['course_title']); ?>
                                                    </div>
                                                <?php endif; ?>
                                                <?php if ($question['tag_names']): ?>
                                                    <div class="question-meta">
                                                        <i class="fas fa-tags me-1"></i><?php echo htmlspecialchars($question['tag_names']); ?>
                                                    </div>
                                                <?php endif; ?>
                                            </div>
                                        </td>
                                        <td>
                                            <span class="badge bg-secondary">
                                                <?php
                                                $typeLabels = [
                                                    'mcq_single' => 'MCQ Single',
                                                    'mcq_multiple' => 'MCQ Multiple',
                                                    'true_false' => 'True/False',
                                                    'essay' => 'Essay',
                                                    'short_answer' => 'Short Answer',
                                                    'matching' => 'Matching',
                                                    'code' => 'Code',
                                                    'fill_blanks' => 'Fill Blanks'
                                                ];
                                                echo $typeLabels[$question['question_type']] ?? $question['question_type'];
                                                ?>
                                            </span>
                                        </td>
                                        <td>
                                            <span class="badge bg-<?php
                                                echo $question['difficulty'] === 'easy' ? 'success' :
                                                     ($question['difficulty'] === 'medium' ? 'warning' : 'danger');
                                            ?>">
                                                <i class="fas fa-<?php
                                                    echo $question['difficulty'] === 'easy' ? 'smile' :
                                                         ($question['difficulty'] === 'medium' ? 'meh' : 'frown');
                                                ?> me-1"></i>
                                                <?php echo ucfirst($question['difficulty']); ?>
                                            </span>
                                        </td>
                                        <td><?php echo htmlspecialchars($question['category_name'] ?? 'Uncategorized'); ?></td>
                                        <td>
                                            <span class="status-badge <?php
                                                echo $question['status'] === 'approved' ? 'active' :
                                                     ($question['status'] === 'pending_review' ? 'pending' :
                                                     ($question['status'] === 'draft' ? 'draft' : 'inactive'));
                                            ?>">
                                                <?php echo ucfirst(str_replace('_', ' ', $question['status'])); ?>
                                            </span>
                                        </td>
                                        <td>
                                            <span class="usage-count">
                                                <i class="fas fa-eye me-1"></i><?php echo $question['usage_count'] ?? 0; ?> times
                                            </span>
                                        </td>
                                        <td>
                                            <div class="table-actions">
                                                <button class="table-action-btn edit" onclick="window.location.href='edit.php?id=<?php echo $question['id']; ?>'" title="Edit">
                                                    <i class="fas fa-edit"></i>
                                                </button>
                                                <button class="table-action-btn" onclick="previewQuestion(<?php echo $question['id']; ?>)" title="Preview">
                                                    <i class="fas fa-eye"></i>
                                                </button>
                                                <button class="table-action-btn delete" onclick="deleteQuestion(<?php echo $question['id']; ?>)" title="Delete">
                                                    <i class="fas fa-trash"></i>
                                                </button>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>
    </div>
</div>

<script>
function previewQuestion(questionId) {
    // Open modal or redirect to preview page
    window.open('preview.php?id=' + questionId, '_blank');
}

function deleteQuestion(questionId) {
    if (confirm('Are you sure you want to delete this question?')) {
        // Implement delete functionality
        alert('Delete functionality to be implemented');
    }
}

function exportQuestions() {
    // Get current filter parameters
    const urlParams = new URLSearchParams(window.location.search);
    const exportUrl = 'export.php?' + urlParams.toString();

    // Create a temporary link and trigger download
    const link = document.createElement('a');
    link.href = exportUrl;
    link.download = 'questions_export.csv';
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
}
</script>

<style>
/* Custom styles for Question Bank */
.admin-container {
    display: flex;
    min-height: 100vh;
    background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
}

.admin-sidebar {
    width: 280px;
    background: linear-gradient(135deg, #1e293b 0%, #334155 100%);
    color: white;
    position: fixed;
    height: 100vh;
    overflow-y: auto;
    box-shadow: 4px 0 15px rgba(0, 0, 0, 0.1);
    z-index: 1000;
}

.sidebar-header {
    padding: 2rem 1.5rem;
    border-bottom: 1px solid #374151;
    text-align: center;
}

.sidebar-header h2 {
    font-size: 1.5rem;
    font-weight: 800;
    margin-bottom: 0.5rem;
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 0.5rem;
}

.sidebar-nav ul {
    list-style: none;
    padding: 0;
    margin: 0;
}

.sidebar-nav li a {
    display: flex;
    align-items: center;
    gap: 0.75rem;
    padding: 1rem 1.5rem;
    color: #cbd5e1;
    text-decoration: none;
    transition: all 0.3s ease;
    font-weight: 500;
    border-left: 3px solid transparent;
}

.sidebar-nav li a:hover,
.sidebar-nav li a.active {
    background: linear-gradient(135deg, #374151 0%, #4b5563 100%);
    color: white;
    border-left-color: #3b82f6;
    transform: translateX(5px);
}

.admin-main {
    margin-left: 280px;
    flex: 1;
    min-height: 100vh;
}

.admin-header {
    background: white;
    padding: 2rem;
    border-bottom: 1px solid #e5e7eb;
    box-shadow: 0 2px 10px rgba(0, 0, 0, 0.05);
}

.admin-header h1 {
    font-size: 2rem;
    font-weight: 800;
    color: #1f2937;
    margin-bottom: 0.5rem;
    display: flex;
    align-items: center;
    gap: 0.75rem;
}

.header-actions {
    display: flex;
    gap: 1rem;
    align-items: center;
}

.admin-content {
    padding: 2rem;
}

.filter-section {
    background: white;
    border-radius: 1rem;
    padding: 2rem;
    margin-bottom: 2rem;
    box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
    border: 1px solid rgba(255, 255, 255, 0.8);
}

.filter-row {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 1.5rem;
    align-items: end;
}

.filter-group label {
    font-weight: 600;
    color: #374151;
    margin-bottom: 0.5rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.filter-actions {
    display: flex;
    gap: 0.75rem;
    align-items: end;
}

.questions-table-container {
    background: white;
    border-radius: 1rem;
    box-shadow: 0 8px 32px rgba(0, 0, 0, 0.1);
    overflow: hidden;
    border: 1px solid rgba(255, 255, 255, 0.8);
}

.table-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 2rem;
    background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
    border-bottom: 1px solid #e5e7eb;
}

.table-header h3 {
    font-size: 1.5rem;
    font-weight: 700;
    color: #1f2937;
    margin: 0;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.table-wrapper {
    overflow-x: auto;
}

.aligned-table {
    width: 100%;
    border-collapse: collapse;
    margin-bottom: 0;
}

.aligned-table thead {
    background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
}

.aligned-table th {
    padding: 1.25rem 1rem;
    text-align: left;
    font-weight: 700;
    color: #374151;
    font-size: 0.875rem;
    text-transform: uppercase;
    letter-spacing: 0.05em;
    border-bottom: 2px solid #e5e7eb;
    white-space: nowrap;
}

.aligned-table td {
    padding: 1.25rem 1rem;
    border-bottom: 1px solid #f3f4f6;
    vertical-align: middle;
}

.aligned-table tbody tr {
    transition: all 0.3s ease;
}

.aligned-table tbody tr:hover {
    background: linear-gradient(135deg, #f8fafc 0%, #f0f4f8 100%);
    transform: scale(1.005);
}

.question-info {
    max-width: 300px;
}

.question-title {
    font-weight: 600;
    color: #1f2937;
    margin-bottom: 0.5rem;
    line-height: 1.4;
}

.question-meta {
    font-size: 0.875rem;
    color: #64748b;
    display: flex;
    align-items: center;
    gap: 0.25rem;
    margin-bottom: 0.25rem;
}

.status-badge {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.5rem 1rem;
    border-radius: 2rem;
    font-size: 0.875rem;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.05em;
}

.status-badge.active {
    background: linear-gradient(135deg, #10b981 0%, #059669 100%);
    color: white;
    box-shadow: 0 4px 15px rgba(16, 185, 129, 0.3);
}

.status-badge.pending {
    background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%);
    color: white;
    box-shadow: 0 4px 15px rgba(245, 158, 11, 0.3);
}

.status-badge.draft {
    background: linear-gradient(135deg, #6b7280 0%, #4b5563 100%);
    color: white;
    box-shadow: 0 4px 15px rgba(107, 114, 128, 0.3);
}

.usage-count {
    font-weight: 600;
    color: #1f2937;
    display: flex;
    align-items: center;
    gap: 0.25rem;
}

.table-actions {
    display: flex;
    gap: 0.5rem;
}

.table-action-btn {
    padding: 0.5rem;
    border-radius: 0.5rem;
    font-size: 0.875rem;
    font-weight: 600;
    text-decoration: none;
    border: none;
    cursor: pointer;
    display: inline-flex;
    align-items: center;
    justify-content: center;
    width: 35px;
    height: 35px;
    transition: all 0.3s ease;
}

.table-action-btn.edit {
    background: linear-gradient(135deg, #3b82f6 0%, #2563eb 100%);
    color: white;
    box-shadow: 0 4px 15px rgba(59, 130, 246, 0.3);
}

.table-action-btn.edit:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(59, 130, 246, 0.4);
}

.table-action-btn {
    background: linear-gradient(135deg, #6b7280 0%, #4b5563 100%);
    color: white;
    box-shadow: 0 4px 15px rgba(107, 114, 128, 0.3);
}

.table-action-btn:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(107, 114, 128, 0.4);
}

.table-action-btn.delete {
    background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%);
    color: white;
    box-shadow: 0 4px 15px rgba(239, 68, 68, 0.3);
}

.table-action-btn.delete:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(239, 68, 68, 0.4);
}

.empty-state {
    text-align: center;
    padding: 4rem 2rem;
    background: white;
    border-radius: 1rem;
    box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
}

.empty-state-icon {
    width: 80px;
    height: 80px;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    margin: 0 auto 2rem;
    color: white;
    font-size: 2rem;
    box-shadow: 0 8px 32px rgba(102, 126, 234, 0.3);
}

.empty-state h3 {
    font-size: 1.5rem;
    font-weight: 700;
    color: #1f2937;
    margin-bottom: 1rem;
}

.empty-state p {
    color: #64748b;
    font-size: 1.1rem;
    margin-bottom: 2rem;
}

.btn-primary {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    border: none;
    border-radius: 0.75rem;
    padding: 0.75rem 1.5rem;
    font-weight: 600;
    transition: all 0.3s ease;
    box-shadow: 0 4px 15px rgba(102, 126, 234, 0.3);
}

.btn-primary:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(102, 126, 234, 0.4);
    background: linear-gradient(135deg, #5a67d8 0%, #6b46c1 100%);
}

.btn-success {
    background: linear-gradient(135deg, #10b981 0%, #059669 100%);
    border: none;
    border-radius: 0.75rem;
    padding: 0.75rem 1.5rem;
    font-weight: 600;
    transition: all 0.3s ease;
    box-shadow: 0 4px 15px rgba(16, 185, 129, 0.3);
}

.btn-success:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(16, 185, 129, 0.4);
    background: linear-gradient(135deg, #059669 0%, #047857 100%);
}

.btn-outline-info {
    background: transparent;
    border: 2px solid #3b82f6;
    color: #3b82f6;
    border-radius: 0.75rem;
    padding: 0.75rem 1.5rem;
    font-weight: 600;
    transition: all 0.3s ease;
}

.btn-outline-info:hover {
    background: #3b82f6;
    color: white;
    transform: translateY(-2px);
    box-shadow: 0 4px 15px rgba(59, 130, 246, 0.3);
}

.btn-outline-secondary {
    background: transparent;
    border: 2px solid #6b7280;
    color: #6b7280;
    border-radius: 0.75rem;
    padding: 0.75rem 1.5rem;
    font-weight: 600;
    transition: all 0.3s ease;
}

.btn-outline-secondary:hover {
    background: #6b7280;
    color: white;
    transform: translateY(-2px);
    box-shadow: 0 4px 15px rgba(107, 114, 128, 0.3);
}

/* Responsive Design */
@media (max-width: 1024px) {
    .admin-sidebar {
        transform: translateX(-100%);
        transition: transform 0.3s ease;
    }

    .admin-main {
        margin-left: 0;
    }

    .filter-row {
        grid-template-columns: 1fr;
    }

    .table-header {
        flex-direction: column;
        align-items: stretch;
        gap: 1rem;
    }

    .aligned-table {
        font-size: 0.875rem;
    }

    .aligned-table th,
    .aligned-table td {
        padding: 0.75rem 0.5rem;
    }

    .question-info {
        max-width: 200px;
    }
}

@media (max-width: 768px) {
    .admin-header {
        padding: 1.5rem;
    }

    .admin-header h1 {
        font-size: 1.5rem;
    }

    .header-actions {
        flex-direction: column;
        align-items: stretch;
        gap: 0.5rem;
    }

    .admin-content {
        padding: 1rem;
    }

    .filter-section {
        padding: 1.5rem;
    }

    .questions-table-container {
        border-radius: 0.75rem;
    }

    .table-header {
        padding: 1.5rem;
    }

    .empty-state {
        padding: 3rem 1rem;
    }

    .empty-state-icon {
        width: 60px;
        height: 60px;
        font-size: 1.5rem;
    }
}
</style>