<?php
require_once '../includes/middleware.php';
require_once '../config/database.php';
require_once '../config/currency.php';
require_once '../includes/functions.php';

// Cart Manager Class (copied from api/cart.php)
class CartManager {
    private $db;
    private $userId;

    public function __construct($db, $userId = null) {
        $this->db = $db;
        $this->userId = $userId;
    }

    public function getCart() {
        if (!$this->userId) {
            return ['items' => [], 'total' => 0.00, 'currency' => 'USD'];
        }

        // Get cart items from database (assuming cart_items table exists)
        $stmt = $this->db->prepare("
            SELECT ci.*, c.title, c.price, c.currency, c.is_free, c.thumbnail, c.max_students
            FROM cart_items ci
            JOIN courses c ON ci.item_id = c.id
            WHERE ci.user_id = ? AND ci.item_type = 'course'
            ORDER BY ci.created_at ASC
        ");
        $stmt->execute([$this->userId]);
        $items = $stmt->fetchAll(PDO::FETCH_ASSOC);

        $total = 0.00;
        $currency = 'USD';

        foreach ($items as &$item) {
            if (!$item['is_free']) {
                $total += floatval($item['price']);
                $currency = $item['currency'];
            }
        }

        return [
            'items' => $items,
            'total' => $total,
            'currency' => $currency,
            'item_count' => count($items)
        ];
    }

    public function removeFromCart($courseId) {
        if (!$this->userId) {
            return ['success' => false, 'error' => 'Authentication required'];
        }

        $stmt = $this->db->prepare("DELETE FROM cart_items WHERE user_id = ? AND item_id = ? AND item_type = 'course'");
        $stmt->execute([$this->userId, $courseId]);

        return ['success' => true, 'message' => 'Course removed from cart'];
    }

    public function clearCart() {
        if (!$this->userId) {
            return ['success' => false, 'error' => 'Authentication required'];
        }

        $stmt = $this->db->prepare("DELETE FROM cart_items WHERE user_id = ? AND item_type = 'course'");
        $stmt->execute([$this->userId]);

        return ['success' => true, 'message' => 'Cart cleared'];
    }

    public function checkout() {
        if (!$this->userId) {
            return ['success' => false, 'error' => 'Authentication required', 'auth_required' => true];
        }

        $cart = $this->getCart();

        if (empty($cart['items'])) {
            return ['success' => false, 'error' => 'Cart is empty'];
        }

        // Separate free and paid items
        $freeItems = [];
        $paidItems = [];

        foreach ($cart['items'] as $item) {
            if ($item['is_free']) {
                $freeItems[] = $item;
            } else {
                $paidItems[] = $item;
            }
        }

        // If cart total is $0.00 (all free items), enroll directly
        if ($cart['total'] == 0.00) {
            return $this->processFreeEnrollment($freeItems);
        }

        // If cart has paid items, return payment details
        return [
            'success' => true,
            'payment_required' => true,
            'cart' => $cart,
            'free_items' => $freeItems,
            'paid_items' => $paidItems
        ];
    }

    private function processFreeEnrollment($freeItems) {
        $enrolledCourses = [];
        $errors = [];

        foreach ($freeItems as $item) {
            try {
                // Check enrollment limits
                if ($item['max_students'] > 0) {
                    $stmt = $this->db->prepare("SELECT COUNT(*) FROM course_enrollments WHERE course_id = ? AND status != 'dropped'");
                    $stmt->execute([$item['item_id']]);
                    $enrollmentCount = $stmt->fetchColumn();

                    if ($enrollmentCount >= $item['max_students']) {
                        $errors[] = "Course '{$item['title']}' is full";
                        continue;
                    }
                }

                // Create enrollment
                $stmt = $this->db->prepare("INSERT INTO course_enrollments (course_id, student_id, payment_status, payment_amount, status, enrollment_date, created_at) VALUES (?, ?, 'paid', 0.00, 'active', NOW(), NOW())");
                $stmt->execute([$item['item_id'], $this->userId]);

                $enrolledCourses[] = $item['title'];

                // Log activity
                if (function_exists('logUserActivity')) {
                    logUserActivity($this->userId, 'course_enrollment', "Enrolled in course ID: {$item['item_id']}");
                }

            } catch (Exception $e) {
                $errors[] = "Failed to enroll in '{$item['title']}': " . $e->getMessage();
            }
        }

        // Clear cart after successful enrollment
        if (!empty($enrolledCourses)) {
            $this->clearCart();
        }

        return [
            'success' => true,
            'enrolled_courses' => $enrolledCourses,
            'errors' => $errors,
            'message' => count($enrolledCourses) . ' course(s) enrolled successfully'
        ];
    }
}

$db = getDB();

// Check if user is logged in
if (!isLoggedIn()) {
    header('Location: login.php?redirect=checkout.php');
    exit;
}

$userId = $_SESSION['user_id'];

// Get cart contents directly
$cartManager = new CartManager($db, $userId);
$cart = $cartManager->getCart();

$page_title = 'Checkout';
include '../includes/header.php';
?>

<!-- Modern Checkout Page -->
<div class="checkout-container">
    <!-- Progress Bar -->
    <div class="checkout-progress">
        <div class="progress-bar">
            <div class="progress-step active">
                <div class="step-icon"><i class="fas fa-shopping-cart"></i></div>
                <span>Cart</span>
            </div>
            <div class="progress-line"></div>
            <div class="progress-step active">
                <div class="step-icon"><i class="fas fa-credit-card"></i></div>
                <span>Checkout</span>
            </div>
            <div class="progress-line"></div>
            <div class="progress-step">
                <div class="step-icon"><i class="fas fa-check"></i></div>
                <span>Complete</span>
            </div>
        </div>
    </div>

    <div class="checkout-main">
        <?php if (empty($cart['items'])): ?>
            <!-- Empty Cart State -->
            <div class="empty-cart">
                <div class="empty-cart-icon">
                    <i class="fas fa-shopping-cart"></i>
                </div>
                <h2>Your cart is empty</h2>
                <p>Add some courses to get started on your learning journey!</p>
                <a href="../courses/catalog.php" class="btn-primary-large">
                    <i class="fas fa-arrow-left me-2"></i>Browse Courses
                </a>
            </div>
        <?php else: ?>
            <div class="checkout-grid">
                <!-- Left Column: Order Summary -->
                <div class="order-summary">
                    <div class="summary-card">
                        <div class="summary-header">
                            <h3><i class="fas fa-receipt me-2"></i>Order Summary</h3>
                        </div>
                        <div class="summary-items">
                            <?php foreach ($cart['items'] as $item): ?>
                                <div class="summary-item">
                                    <div class="item-info">
                                        <?php if ($item['thumbnail']): ?>
                                            <img src="<?php echo htmlspecialchars($item['thumbnail']); ?>" alt="" class="item-thumbnail">
                                        <?php endif; ?>
                                        <div class="item-details">
                                            <h4><?php echo htmlspecialchars($item['title']); ?></h4>
                                            <?php if ($item['is_free']): ?>
                                                <span class="badge-free">Free</span>
                                            <?php else: ?>
                                                <span class="item-price"><?php echo formatCurrency($item['price'], $item['currency']); ?></span>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                    <button class="btn-remove" onclick="removeFromCart(<?php echo $item['item_id']; ?>)">
                                        <i class="fas fa-trash"></i>
                                    </button>
                                </div>
                            <?php endforeach; ?>
                        </div>
                        <div class="summary-total">
                            <div class="total-row">
                                <span>Total</span>
                                <span class="total-amount"><?php echo formatCurrency($cart['total'], $cart['currency']); ?></span>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Right Column: Payment Form -->
                <div class="payment-section">
                    <?php if ($cart['total'] == 0.00): ?>
                        <!-- Free Enrollment -->
                        <div class="payment-card">
                            <div class="payment-header">
                                <h3><i class="fas fa-graduation-cap me-2"></i>Free Enrollment</h3>
                            </div>
                            <div class="free-notice">
                                <i class="fas fa-check-circle"></i>
                                <p>All items in your cart are free! Click below to enroll immediately.</p>
                            </div>
                            <button id="enrollNowBtn" class="btn-primary-large" onclick="processFreeEnrollment()">
                                <i class="fas fa-graduation-cap me-2"></i>Enroll Now
                            </button>
                        </div>
                    <?php else: ?>
                        <!-- Paid Checkout -->
                        <div class="payment-card">
                            <div class="payment-header">
                                <h3><i class="fas fa-credit-card me-2"></i>Payment Details</h3>
                            </div>

                            <form id="paymentForm">
                                <input type="hidden" name="item_type" value="cart">
                                <input type="hidden" name="cart_total" value="<?php echo $cart['total']; ?>">
                                <input type="hidden" name="currency" value="<?php echo $cart['currency']; ?>">

                                <!-- Discount Code -->
                                <div class="form-group">
                                    <label>Discount Code <span class="optional">(Optional)</span></label>
                                    <input type="text" class="form-input" name="discount_code" placeholder="Enter discount code">
                                    <div id="discountMessage" class="form-hint"></div>
                                </div>

                                <!-- Payment Method -->
                                <div class="form-group">
                                    <label>Payment Method</label>
                                    <div class="payment-methods">
                                        <label class="payment-option">
                                            <input type="radio" name="gateway" value="paynow" checked>
                                            <div class="payment-content">
                                                <i class="fas fa-mobile-alt"></i>
                                                <span>Paynow (Mobile Money)</span>
                                            </div>
                                        </label>
                                        <label class="payment-option">
                                            <input type="radio" name="gateway" value="paypal">
                                            <div class="payment-content">
                                                <i class="fab fa-paypal"></i>
                                                <span>PayPal</span>
                                            </div>
                                        </label>
                                    </div>
                                </div>

                                <!-- Credit Card Form (for PayPal) -->
                                <div id="creditCardForm" class="credit-card-section" style="display: none;">
                                    <h4>Credit Card Information</h4>
                                    <div class="card-form">
                                        <div class="form-group">
                                            <label>Card Number</label>
                                            <input type="text" class="form-input" name="card_number" placeholder="1234 5678 9012 3456">
                                        </div>
                                        <div class="form-row">
                                            <div class="form-group">
                                                <label>Expiry Date</label>
                                                <input type="text" class="form-input" name="expiry_date" placeholder="MM/YY">
                                            </div>
                                            <div class="form-group">
                                                <label>CVV</label>
                                                <input type="text" class="form-input" name="cvv" placeholder="123">
                                            </div>
                                        </div>
                                        <div class="form-group">
                                            <label>Cardholder Name</label>
                                            <input type="text" class="form-input" name="cardholder_name" placeholder="John Doe">
                                        </div>
                                    </div>
                                </div>

                                <!-- Final Amount -->
                                <div id="finalAmount" class="final-amount">
                                    <div class="amount-display">
                                        <span>Total Amount</span>
                                        <span class="amount"><?php echo formatCurrency($cart['total'], $cart['currency']); ?></span>
                                    </div>
                                </div>

                                <!-- PayNow Button (shown when PayNow is selected) -->
                                <div id="paynowSection" style="display: none; text-align: center; margin-bottom: 1.5rem;">
                                    <a href='https://www.paynow.co.zw/Payment/Link/?q=c2VhcmNoPW11dGFsZXh6dyU0MGdtYWlsLmNvbSZhbW91bnQ9MS4wMCZyZWZlcmVuY2U9TUEwMDAwMDAmbD0x' target='_blank'>
                                        <img src='https://www.paynow.co.zw/Content/Buttons/Medium_buttons/button_pay-now_medium.png' style='border:0' />
                                    </a>
                                </div>

                                <!-- Pay Button -->
                                <button type="submit" class="btn-primary-large" id="payButton">
                                    <i class="fas fa-lock me-2"></i>
                                    Pay <?php echo formatCurrency($cart['total'], $cart['currency']); ?>
                                </button>
                            </form>
                        </div>
                    <?php endif; ?>

                    <!-- Trust Badges -->
                    <div class="trust-badges">
                        <div class="badge">
                            <i class="fas fa-shield-alt"></i>
                            <span>Secure Payment</span>
                        </div>
                        <div class="badge">
                            <i class="fas fa-lock"></i>
                            <span>SSL Encrypted</span>
                        </div>
                        <div class="badge">
                            <i class="fas fa-undo"></i>
                            <span>Money Back Guarantee</span>
                        </div>
                    </div>
                </div>
            </div>
        <?php endif; ?>
    </div>
</div>

<script>
// Toggle forms based on payment method
document.querySelectorAll('input[name="gateway"]').forEach(radio => {
    radio.addEventListener('change', function() {
        const creditCardForm = document.getElementById('creditCardForm');
        const paynowSection = document.getElementById('paynowSection');
        if (this.value === 'paypal') {
            creditCardForm.style.display = 'block';
            paynowSection.style.display = 'none';
        } else if (this.value === 'paynow') {
            creditCardForm.style.display = 'none';
            paynowSection.style.display = 'block';
        } else {
            creditCardForm.style.display = 'none';
            paynowSection.style.display = 'none';
        }
    });
});

// Handle discount code validation
document.querySelector('input[name="discount_code"]')?.addEventListener('blur', function() {
    validateDiscountCode(this.value, <?php echo $cart['total']; ?>, '<?php echo $cart['currency']; ?>');
});

function validateDiscountCode(code, originalPrice, currency) {
    if (!code.trim()) {
        document.getElementById('discountMessage').innerHTML = '';
        updateFinalAmount(originalPrice, 0, currency);
        return;
    }

    document.getElementById('discountMessage').innerHTML = '<small class="text-muted">Validating discount code...</small>';

    // Simulate API call
    setTimeout(() => {
        if (code.toUpperCase().startsWith('TEST')) {
            const discountAmount = originalPrice * 0.1;
            document.getElementById('discountMessage').innerHTML = '<small class="text-success">10% discount applied!</small>';
            updateFinalAmount(originalPrice, discountAmount, currency);
        } else {
            document.getElementById('discountMessage').innerHTML = '<small class="text-danger">Invalid discount code</small>';
            updateFinalAmount(originalPrice, 0, currency);
        }
    }, 500);
}

function updateFinalAmount(originalPrice, discountAmount, currency) {
    const finalAmount = originalPrice - discountAmount;
    document.getElementById('finalAmount').innerHTML = `<strong>Final Amount: ${currency} ${finalAmount.toFixed(2)}</strong>`;
    document.getElementById('payButton').innerHTML = `<i class="fas fa-credit-card me-2"></i>Pay ${currency} ${finalAmount.toFixed(2)}`;
}

function removeFromCart(courseId) {
    if (confirm('Remove this course from cart?')) {
        fetch('../api/cart.php?course_id=' + courseId, {
            method: 'DELETE'
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                location.reload();
            } else {
                alert('Error: ' + data.error);
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('An error occurred. Please try again.');
        });
    }
}

function processFreeEnrollment() {
    const btn = document.getElementById('enrollNowBtn');
    const originalText = btn.innerHTML;

    btn.disabled = true;
    btn.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>Enrolling...';

    fetch('../api/cart.php', {
        method: 'PUT',
        headers: {
            'Content-Type': 'application/json',
        }
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            alert('Successfully enrolled in ' + data.enrolled_courses.length + ' course(s)!');
            window.location.href = '../profile.php';
        } else {
            alert('Error: ' + (data.error || 'Enrollment failed'));
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('An error occurred. Please try again.');
    })
    .finally(() => {
        btn.disabled = false;
        btn.innerHTML = originalText;
    });
}

// Handle payment form submission
document.getElementById('paymentForm')?.addEventListener('submit', function(e) {
    e.preventDefault();
    processCartPayment(new FormData(this));
});

function processCartPayment(formData) {
    const payButton = document.getElementById('payButton');
    const originalText = payButton.innerHTML;

    payButton.disabled = true;
    payButton.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>Processing...';

    fetch('../api/payments.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            if (data.redirect_url) {
                window.location.href = data.redirect_url;
            } else {
                alert('Payment initiated successfully!');
                window.location.href = 'dashboard.php';
            }
        } else {
            showPaymentError(data.error || 'Unknown error');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('An error occurred. Please try again.');
    })
    .finally(() => {
        payButton.disabled = false;
        payButton.innerHTML = originalText;
    });
}
</script>

<style>
/* Modern Checkout Styles */
.checkout-container {
    max-width: 1200px;
    margin: 0 auto;
    padding: 2rem 1rem;
    background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
    min-height: calc(100vh - 160px);
}

.checkout-progress {
    margin-bottom: 2rem;
}

.progress-bar {
    display: flex;
    align-items: center;
    justify-content: center;
    max-width: 600px;
    margin: 0 auto;
}

.progress-step {
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 0.5rem;
    opacity: 0.5;
    transition: all 0.3s ease;
}

.progress-step.active {
    opacity: 1;
}

.step-icon {
    width: 50px;
    height: 50px;
    border-radius: 50%;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-size: 1.2rem;
    box-shadow: 0 4px 15px rgba(102, 126, 234, 0.3);
}

.progress-step.active .step-icon {
    background: linear-gradient(135deg, #4f46e5 0%, #7c3aed 100%);
    animation: pulse 2s infinite;
}

.progress-line {
    width: 80px;
    height: 2px;
    background: #e5e7eb;
    margin: 0 1rem;
}

.progress-step.active + .progress-line {
    background: linear-gradient(90deg, #4f46e5, #7c3aed);
}

.checkout-main {
    max-width: 1000px;
    margin: 0 auto;
}

.checkout-grid {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 2rem;
    align-items: start;
}

/* Empty Cart */
.empty-cart {
    text-align: center;
    padding: 4rem 2rem;
    background: white;
    border-radius: 1rem;
    box-shadow: 0 10px 25px rgba(0, 0, 0, 0.1);
    grid-column: 1 / -1;
}

.empty-cart-icon {
    font-size: 4rem;
    color: #cbd5e1;
    margin-bottom: 1rem;
}

.empty-cart h2 {
    color: #1f2937;
    margin-bottom: 1rem;
    font-size: 1.8rem;
}

.empty-cart p {
    color: #64748b;
    margin-bottom: 2rem;
    font-size: 1.1rem;
}

/* Order Summary */
.order-summary {
    /* Removed sticky positioning */
}

.summary-card {
    background: white;
    border-radius: 1rem;
    box-shadow: 0 10px 25px rgba(0, 0, 0, 0.1);
    overflow: hidden;
    border: 1px solid #e5e7eb;
}

.summary-header {
    background: linear-gradient(135deg, #4f46e5 0%, #7c3aed 100%);
    color: white;
    padding: 1.5rem;
}

.summary-header h3 {
    margin: 0;
    font-size: 1.3rem;
    font-weight: 600;
}

.summary-items {
    padding: 1.5rem;
}

.summary-item {
    display: flex;
    align-items: center;
    justify-content: space-between;
    padding: 1rem 0;
    border-bottom: 1px solid #f1f5f9;
}

.summary-item:last-child {
    border-bottom: none;
}

.item-info {
    display: flex;
    align-items: center;
    gap: 1rem;
    flex: 1;
}

.item-thumbnail {
    width: 60px;
    height: 40px;
    object-fit: cover;
    border-radius: 0.5rem;
    border: 2px solid #e5e7eb;
}

.item-details h4 {
    margin: 0 0 0.25rem 0;
    font-size: 1rem;
    font-weight: 600;
    color: #1f2937;
}

.badge-free {
    background: #dcfce7;
    color: #166534;
    padding: 0.25rem 0.75rem;
    border-radius: 0.5rem;
    font-size: 0.8rem;
    font-weight: 600;
}

.item-price {
    color: #4f46e5;
    font-weight: 600;
    font-size: 1rem;
}

.btn-remove {
    background: none;
    border: none;
    color: #dc2626;
    cursor: pointer;
    padding: 0.5rem;
    border-radius: 0.5rem;
    transition: all 0.3s ease;
}

.btn-remove:hover {
    background: #fef2f2;
    transform: scale(1.1);
}

.summary-total {
    background: #f8fafc;
    padding: 1.5rem;
    border-top: 1px solid #e5e7eb;
}

.total-row {
    display: flex;
    justify-content: space-between;
    align-items: center;
    font-size: 1.2rem;
    font-weight: 600;
    color: #1f2937;
}

.total-amount {
    color: #4f46e5;
    font-size: 1.4rem;
}

/* Payment Section */
.payment-section {
    display: flex;
    flex-direction: column;
    gap: 2rem;
}

.payment-card {
    background: white;
    border-radius: 1rem;
    box-shadow: 0 10px 25px rgba(0, 0, 0, 0.1);
    overflow: hidden;
    border: 1px solid #e5e7eb;
}

.payment-header {
    background: linear-gradient(135deg, #10b981 0%, #059669 100%);
    color: white;
    padding: 1.5rem;
}

.payment-header h3 {
    margin: 0;
    font-size: 1.3rem;
    font-weight: 600;
}

.free-notice {
    padding: 2rem;
    text-align: center;
    background: #f0fdf4;
    border-bottom: 1px solid #bbf7d0;
}

.free-notice i {
    font-size: 3rem;
    color: #16a34a;
    margin-bottom: 1rem;
}

.free-notice p {
    color: #166534;
    font-size: 1.1rem;
    margin: 0;
}

/* Form Styles */
.form-group {
    margin-bottom: 1.5rem;
}

.form-group label {
    display: block;
    margin-bottom: 0.5rem;
    font-weight: 600;
    color: #374151;
    font-size: 0.95rem;
}

.optional {
    font-weight: 400;
    color: #6b7280;
    font-size: 0.85rem;
}

.form-input {
    width: 100%;
    padding: 0.875rem 1rem;
    border: 2px solid #e5e7eb;
    border-radius: 0.5rem;
    font-size: 1rem;
    transition: all 0.3s ease;
    background: white;
}

.form-input:focus {
    outline: none;
    border-color: #4f46e5;
    box-shadow: 0 0 0 3px rgba(79, 70, 229, 0.1);
}

.form-hint {
    margin-top: 0.25rem;
    font-size: 0.85rem;
}

.form-row {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 1rem;
}

/* Payment Methods */
.payment-methods {
    display: flex;
    flex-direction: column;
    gap: 1rem;
}

.payment-option {
    display: flex;
    align-items: center;
    cursor: pointer;
    padding: 1rem;
    border: 2px solid #e5e7eb;
    border-radius: 0.5rem;
    transition: all 0.3s ease;
    background: white;
}

.payment-option:hover {
    border-color: #4f46e5;
    box-shadow: 0 4px 12px rgba(79, 70, 229, 0.1);
}

.payment-option input[type="radio"] {
    margin-right: 1rem;
    accent-color: #4f46e5;
}

.payment-content {
    display: flex;
    align-items: center;
    gap: 0.75rem;
    flex: 1;
}

.payment-content i {
    font-size: 1.5rem;
    color: #4f46e5;
}

.payment-content span {
    font-weight: 600;
    color: #1f2937;
}

/* Credit Card Form */
.credit-card-section {
    padding: 0 1.5rem 1.5rem;
}

.credit-card-section h4 {
    margin-bottom: 1rem;
    color: #1f2937;
    font-size: 1.1rem;
}

.card-form {
    background: #f8fafc;
    padding: 1.5rem;
    border-radius: 0.5rem;
    border: 1px solid #e5e7eb;
}

/* Final Amount */
.final-amount {
    padding: 1.5rem;
    background: #f0f9ff;
    border: 1px solid #bae6fd;
    border-radius: 0.5rem;
    margin-bottom: 1.5rem;
}

.amount-display {
    display: flex;
    justify-content: space-between;
    align-items: center;
    font-size: 1.2rem;
    font-weight: 600;
    color: #1f2937;
}

.amount {
    color: #4f46e5;
    font-size: 1.4rem;
}

/* Buttons */
.btn-primary-large {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    gap: 0.5rem;
    width: 100%;
    padding: 1rem 2rem;
    background: linear-gradient(135deg, #4f46e5 0%, #7c3aed 100%);
    color: white;
    border: none;
    border-radius: 0.5rem;
    font-size: 1.1rem;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s ease;
    text-decoration: none;
    box-shadow: 0 4px 15px rgba(79, 70, 229, 0.3);
}

.btn-primary-large:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(79, 70, 229, 0.4);
}

.btn-primary-large:active {
    transform: translateY(0);
}

/* Trust Badges */
.trust-badges {
    display: flex;
    justify-content: center;
    gap: 2rem;
    margin-top: 2rem;
}

.badge {
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 0.5rem;
    padding: 1rem;
    background: white;
    border-radius: 0.5rem;
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
    border: 1px solid #e5e7eb;
    min-width: 120px;
}

.badge i {
    font-size: 1.5rem;
    color: #10b981;
}

.badge span {
    font-size: 0.85rem;
    font-weight: 600;
    color: #374151;
    text-align: center;
}

/* Animations */
@keyframes pulse {
    0%, 100% { transform: scale(1); }
    50% { transform: scale(1.05); }
}

@keyframes fadeInUp {
    from {
        opacity: 0;
        transform: translateY(30px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.checkout-container {
    animation: fadeInUp 0.6s ease-out;
}

/* Responsive Design */
@media (max-width: 768px) {
    .checkout-container {
        padding: 1rem;
    }

    .checkout-grid {
        grid-template-columns: 1fr;
        gap: 1.5rem;
    }

    .order-summary {
        position: static;
    }

    .progress-bar {
        flex-direction: column;
        gap: 1rem;
    }

    .progress-line {
        width: 2px;
        height: 40px;
        margin: 0;
    }

    .trust-badges {
        flex-direction: column;
        gap: 1rem;
    }

    .badge {
        flex-direction: row;
        min-width: auto;
    }

    .form-row {
        grid-template-columns: 1fr;
    }

    .checkout-progress {
        margin-bottom: 1.5rem;
    }
}

@media (max-width: 480px) {
    .checkout-container {
        padding: 0.5rem;
    }

    .empty-cart {
        padding: 2rem 1rem;
    }

    .empty-cart-icon {
        font-size: 3rem;
    }

    .summary-items,
    .summary-total {
        padding: 1rem;
    }

    .payment-card {
        margin: 0 -0.5rem;
    }
}
</style>

<?php include '../includes/footer.php'; ?>