<?php
require_once 'BaseAIService.php';

class TranslationService extends BaseAIService {

    protected function isFeatureEnabled() {
        return isAIFeatureEnabled('translation');
    }

    protected function getFeatureName() {
        return 'translation';
    }

    protected function validateInput($data) {
        if (!isset($data['content']) || empty($data['content'])) {
            throw new Exception('Content is required for translation');
        }

        if (!isset($data['target_language']) || empty($data['target_language'])) {
            throw new Exception('Target language is required');
        }

        $config = getAIConfig('translation');
        if (!in_array($data['target_language'], $config['supported_languages'])) {
            throw new Exception('Unsupported target language');
        }

        return true;
    }

    public function translateContent($content, $targetLanguage, $context = 'educational') {
        $this->validateInput([
            'content' => $content,
            'target_language' => $targetLanguage
        ]);

        $prompt = $this->buildTranslationPrompt($content, $targetLanguage, $context);

        $messages = [
            [
                'role' => 'system',
                'content' => 'You are a professional translator specializing in educational content. Provide accurate, culturally appropriate translations.'
            ],
            [
                'role' => 'user',
                'content' => $prompt
            ]
        ];

        $response = $this->makeRequest($messages);

        return [
            'original_content' => $content,
            'translated_content' => trim($response->choices[0]->message->content),
            'target_language' => $targetLanguage,
            'context' => $context
        ];
    }

    private function buildTranslationPrompt($content, $targetLanguage, $context) {
        $languageNames = [
            'es' => 'Spanish',
            'fr' => 'French',
            'de' => 'German',
            'zh' => 'Chinese',
            'ja' => 'Japanese',
            'ar' => 'Arabic',
            'pt' => 'Portuguese',
            'ru' => 'Russian',
            'hi' => 'Hindi'
        ];

        $targetName = $languageNames[$targetLanguage] ?? $targetLanguage;

        $prompt = "Translate the following {$context} content to {$targetName}.\n\n";
        $prompt .= "Content:\n{$content}\n\n";
        $prompt .= "Guidelines:\n";
        $prompt .= "- Maintain educational tone and terminology\n";
        $prompt .= "- Ensure cultural appropriateness\n";
        $prompt .= "- Preserve technical accuracy\n";
        $prompt .= "- Keep the translation natural and readable\n";
        $prompt .= "- Maintain the original meaning and intent";

        return $prompt;
    }

    public function batchTranslate($contents, $targetLanguage, $context = 'educational') {
        $results = [];

        foreach ($contents as $content) {
            try {
                $results[] = $this->translateContent($content, $targetLanguage, $context);
            } catch (Exception $e) {
                $results[] = [
                    'original_content' => $content,
                    'error' => $e->getMessage(),
                    'status' => 'error'
                ];
            }
        }

        return $results;
    }
}
?>