<?php
require_once '../../config/database.php';
require_once '../../config/auth.php';
require_once '../../config/rbac.php';
$lms_config = require '../../config/lms.php';
require_once '../../includes/functions.php';
require_once '../../includes/middleware.php';

// Check if user is instructor or admin
anyRoleMiddleware(['instructor', 'admin']);

$user = getUserContext();
$db = getDB();
$userId = $_SESSION['user_id'];

$page_title = 'Import Questions - ' . $lms_config['lms']['site_name'];
?>

<div class="admin-container">
    <div class="admin-sidebar">
        <div class="sidebar-header">
            <h2><i class="fas fa-graduation-cap me-2"></i><?php echo htmlspecialchars($lms_config['lms']['site_name']); ?></h2>
            <p><?php echo htmlspecialchars($lms_config['lms']['site_name']); ?></p>
        </div>

        <nav class="sidebar-nav">
            <ul>
                <li><a href="../dashboard.php"><i class="fas fa-tachometer-alt me-2"></i>Dashboard</a></li>
                <li><a href="../courses/"><i class="fas fa-book me-2"></i>My Courses</a></li>
                <li><a href="../courses/new.php"><i class="fas fa-plus me-2"></i>Create Course</a></li>
                <li><a href="index.php" class="active"><i class="fas fa-question-circle me-2"></i>Question Bank</a></li>
                <li><a href="../exams/"><i class="fas fa-file-alt me-2"></i>Exams</a></li>
                <li><a href="../resources.php"><i class="fas fa-folder me-2"></i>Resources</a></li>
                <li><a href="../students.php"><i class="fas fa-users me-2"></i>Students</a></li>
                <li><a href="../assignments.php"><i class="fas fa-tasks me-2"></i>Assignments</a></li>
                <li><a href="/lms/instructor/grading/index.php"><i class="fas fa-chart-line me-2"></i>Grades</a></li>
                <li><a href="../../logout.php"><i class="fas fa-sign-out-alt me-2"></i>Logout</a></li>
            </ul>
        </nav>
    </div>

    <div class="admin-main">
        <div class="admin-header">
            <div class="d-flex justify-content-between align-items-center">
                <div>
                    <h1><i class="fas fa-upload text-warning me-3"></i>Import Questions</h1>
                    <p class="text-muted mb-0">Bulk import questions from CSV files to expand your question bank</p>
                </div>
                <a href="index.php" class="btn btn-outline-secondary btn-lg">
                    <i class="fas fa-arrow-left me-2"></i>Back to Questions
                </a>
            </div>
        </div>

        <div class="admin-content">

            <div class="import-layout">
                <!-- Main Import Form -->
                <div class="import-main-form">
                    <div class="form-card">
                        <div class="form-card-header">
                            <h3><i class="fas fa-file-csv me-2"></i>CSV Import</h3>
                            <p>Upload a CSV file containing your questions for bulk import</p>
                        </div>
                        <div class="form-card-body">
                            <form method="POST" enctype="multipart/form-data" action="process_import.php">
                                <div class="form-group">
                                    <label for="csv_file"><i class="fas fa-file-upload me-1"></i>Select CSV File <span class="text-danger">*</span></label>
                                    <input type="file" class="form-control" id="csv_file" name="csv_file"
                                            accept=".csv" required>
                                    <div class="form-text">
                                        CSV file should contain columns: title, content, question_type, difficulty, points, category_name, explanation
                                    </div>
                                </div>

                                <div class="form-group">
                                    <label for="default_course"><i class="fas fa-book me-1"></i>Default Course (Optional)</label>
                                    <select class="form-select" id="default_course" name="default_course">
                                        <option value="">No default course</option>
                                        <?php
                                        $courses = $db->prepare("SELECT id, title FROM courses WHERE instructor_id = ? ORDER BY title", [$userId])->fetchAll();
                                        foreach ($courses as $course): ?>
                                            <option value="<?php echo $course['id']; ?>">
                                                <?php echo htmlspecialchars($course['title']); ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>

                                <div class="form-group">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" id="skip_duplicates" name="skip_duplicates" checked>
                                        <label class="form-check-label" for="skip_duplicates">
                                            <i class="fas fa-shield-alt me-1"></i>Skip duplicate questions (based on title)
                                        </label>
                                    </div>
                                </div>

                                <div class="form-group">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" id="auto_approve" name="auto_approve">
                                        <label class="form-check-label" for="auto_approve">
                                            <i class="fas fa-check-circle me-1"></i>Auto-approve imported questions
                                        </label>
                                    </div>
                                </div>

                                <div class="form-actions">
                                    <button type="submit" class="btn btn-primary btn-lg">
                                        <i class="fas fa-upload me-2"></i>Import Questions
                                    </button>
                                    <a href="index.php" class="btn btn-outline-secondary">
                                        <i class="fas fa-times me-1"></i>Cancel
                                    </a>
                                </div>
                            </form>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Import Information Sidebar -->
                <div class="import-sidebar">
                    <div class="info-card">
                        <div class="info-card-header">
                            <h3><i class="fas fa-info-circle me-2"></i>CSV Format Guide</h3>
                        </div>
                        <div class="info-card-body">
                            <div class="info-section">
                                <h4><i class="fas fa-asterisk text-danger me-1"></i>Required Columns:</h4>
                                <ul class="info-list">
                                    <li><code>title</code> - Question title</li>
                                    <li><code>content</code> - Question content</li>
                                    <li><code>question_type</code> - Type of question</li>
                                </ul>
                            </div>

                            <div class="info-section">
                                <h4><i class="fas fa-plus text-info me-1"></i>Optional Columns:</h4>
                                <ul class="info-list">
                                    <li><code>difficulty</code> - easy/medium/hard</li>
                                    <li><code>points</code> - Point value</li>
                                    <li><code>category_name</code> - Question category</li>
                                    <li><code>explanation</code> - Answer explanation</li>
                                    <li><code>hints</code> - Comma-separated hints</li>
                                </ul>
                            </div>

                            <div class="info-section">
                                <h4><i class="fas fa-list me-1"></i>Question Types:</h4>
                                <div class="question-types">
                                    <div class="type-group">
                                        <strong>Multiple Choice:</strong> mcq_single, mcq_multiple
                                    </div>
                                    <div class="type-group">
                                        <strong>Simple:</strong> true_false
                                    </div>
                                    <div class="type-group">
                                        <strong>Open-ended:</strong> essay, short_answer
                                    </div>
                                    <div class="type-group">
                                        <strong>Advanced:</strong> matching, code, fill_blanks
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <div class="sample-card">
                        <div class="sample-card-header">
                            <h3><i class="fas fa-file-code me-2"></i>Sample CSV</h3>
                        </div>
                        <div class="sample-card-body">
                            <pre class="code-sample"><code>title,content,question_type,difficulty,points,category_name,explanation
"What is PHP?","PHP is a server-side scripting language","mcq_single","easy",1,"Programming","PHP stands for PHP: Hypertext Preprocessor"
"What does HTML stand for?","HTML stands for...","fill_blanks","medium",2,"Web Development","HyperText Markup Language"</code></pre>
                            <a href="sample_questions.csv" class="btn btn-outline-primary btn-sm" download>
                                <i class="fas fa-download me-1"></i>Download Sample
                            </a>
                        </div>
                    </div>
                </div>
            </div>
    </div>
</div>

<style>
/* Custom styles for Import Questions page */
.admin-container {
    display: flex;
    min-height: 100vh;
    background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
}

.admin-sidebar {
    width: 280px;
    background: linear-gradient(135deg, #1e293b 0%, #334155 100%);
    color: white;
    position: fixed;
    height: 100vh;
    overflow-y: auto;
    box-shadow: 4px 0 15px rgba(0, 0, 0, 0.1);
    z-index: 1000;
}

.sidebar-header {
    padding: 2rem 1.5rem;
    border-bottom: 1px solid #374151;
    text-align: center;
}

.sidebar-header h2 {
    font-size: 1.5rem;
    font-weight: 800;
    margin-bottom: 0.5rem;
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 0.5rem;
}

.sidebar-nav ul {
    list-style: none;
    padding: 0;
    margin: 0;
}

.sidebar-nav li a {
    display: flex;
    align-items: center;
    gap: 0.75rem;
    padding: 1rem 1.5rem;
    color: #cbd5e1;
    text-decoration: none;
    transition: all 0.3s ease;
    font-weight: 500;
    border-left: 3px solid transparent;
}

.sidebar-nav li a:hover,
.sidebar-nav li a.active {
    background: linear-gradient(135deg, #374151 0%, #4b5563 100%);
    color: white;
    border-left-color: #3b82f6;
    transform: translateX(5px);
}

.admin-main {
    margin-left: 280px;
    flex: 1;
    min-height: 100vh;
}

.admin-header {
    background: white;
    padding: 2rem;
    border-bottom: 1px solid #e5e7eb;
    box-shadow: 0 2px 10px rgba(0, 0, 0, 0.05);
}

.admin-header h1 {
    font-size: 2rem;
    font-weight: 800;
    color: #1f2937;
    margin-bottom: 0.5rem;
    display: flex;
    align-items: center;
    gap: 0.75rem;
}

.admin-content {
    padding: 2rem;
}

.import-layout {
    display: grid;
    grid-template-columns: 2fr 1fr;
    gap: 2rem;
    align-items: start;
}

.import-main-form {
    display: flex;
    flex-direction: column;
    gap: 2rem;
}

.import-sidebar {
    position: sticky;
    top: 2rem;
    display: flex;
    flex-direction: column;
    gap: 2rem;
}

.form-card {
    background: white;
    border-radius: 1rem;
    box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
    border: 1px solid rgba(255, 255, 255, 0.8);
    overflow: hidden;
}

.form-card-header {
    padding: 2rem;
    background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
    border-bottom: 1px solid #e5e7eb;
}

.form-card-header h3 {
    font-size: 1.25rem;
    font-weight: 700;
    color: #1f2937;
    margin-bottom: 0.5rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.form-card-header p {
    color: #64748b;
    margin: 0;
    font-size: 0.9rem;
}

.form-card-body {
    padding: 2rem;
}

.form-group {
    margin-bottom: 1.5rem;
}

.form-group label {
    display: block;
    font-weight: 600;
    color: #374151;
    margin-bottom: 0.5rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.form-control {
    width: 100%;
    padding: 0.75rem 1rem;
    border: 2px solid #e5e7eb;
    border-radius: 0.75rem;
    font-size: 0.875rem;
    transition: all 0.3s ease;
    background: white;
}

.form-control:focus {
    outline: none;
    border-color: #667eea;
    box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
}

.form-select {
    width: 100%;
    padding: 0.75rem 1rem;
    border: 2px solid #e5e7eb;
    border-radius: 0.75rem;
    font-size: 0.875rem;
    transition: all 0.3s ease;
    background: white;
}

.form-select:focus {
    outline: none;
    border-color: #667eea;
    box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
}

.form-text {
    color: #64748b;
    font-size: 0.875rem;
    margin-top: 0.25rem;
}

.form-check {
    display: flex;
    align-items: center;
    gap: 0.75rem;
    margin-bottom: 1rem;
}

.form-check-input {
    width: 1.25rem;
    height: 1.25rem;
    border: 2px solid #e5e7eb;
    border-radius: 0.375rem;
    background: white;
    cursor: pointer;
    transition: all 0.3s ease;
}

.form-check-input:checked {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    border-color: #667eea;
}

.form-check-label {
    color: #374151;
    font-weight: 500;
    cursor: pointer;
}

.form-actions {
    display: flex;
    gap: 1rem;
    justify-content: center;
    margin-top: 2rem;
    padding-top: 2rem;
    border-top: 1px solid #e5e7eb;
}

.btn {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.75rem 1.5rem;
    font-weight: 600;
    text-decoration: none;
    border-radius: 0.75rem;
    border: none;
    cursor: pointer;
    transition: all 0.3s ease;
    font-size: 0.875rem;
}

.btn-primary {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    box-shadow: 0 4px 15px rgba(102, 126, 234, 0.3);
}

.btn-primary:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(102, 126, 234, 0.4);
    background: linear-gradient(135deg, #5a67d8 0%, #6b46c1 100%);
}

.btn-outline-secondary {
    background: transparent;
    border: 2px solid #6b7280;
    color: #6b7280;
}

.btn-outline-secondary:hover {
    background: #6b7280;
    color: white;
    transform: translateY(-2px);
    box-shadow: 0 4px 15px rgba(107, 114, 128, 0.3);
}

.btn-outline-primary {
    background: transparent;
    border: 2px solid #667eea;
    color: #667eea;
}

.btn-outline-primary:hover {
    background: #667eea;
    color: white;
    transform: translateY(-2px);
    box-shadow: 0 4px 15px rgba(102, 126, 234, 0.3);
}

.btn-lg {
    padding: 0.875rem 2rem;
    font-size: 1rem;
}

.btn-sm {
    padding: 0.5rem 1rem;
    font-size: 0.8rem;
}

.text-danger {
    color: #dc2626 !important;
}

.text-muted {
    color: #64748b !important;
}

.info-card,
.sample-card {
    background: white;
    border-radius: 1rem;
    box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
    border: 1px solid rgba(255, 255, 255, 0.8);
    overflow: hidden;
}

.info-card-header,
.sample-card-header {
    padding: 1.5rem;
    background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
    border-bottom: 1px solid #e5e7eb;
}

.info-card-header h3,
.sample-card-header h3 {
    font-size: 1.1rem;
    font-weight: 700;
    color: #1f2937;
    margin: 0;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.info-card-body,
.sample-card-body {
    padding: 1.5rem;
}

.info-section {
    margin-bottom: 1.5rem;
}

.info-section:last-child {
    margin-bottom: 0;
}

.info-section h4 {
    font-size: 0.9rem;
    font-weight: 600;
    color: #374151;
    margin-bottom: 0.75rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.info-list {
    list-style: none;
    padding: 0;
    margin: 0;
}

.info-list li {
    color: #64748b;
    font-size: 0.875rem;
    margin-bottom: 0.5rem;
    padding-left: 1rem;
    position: relative;
}

.info-list li:before {
    content: '•';
    color: #667eea;
    font-weight: bold;
    position: absolute;
    left: 0;
}

.question-types {
    display: flex;
    flex-direction: column;
    gap: 0.75rem;
}

.type-group {
    font-size: 0.875rem;
    color: #64748b;
}

.type-group strong {
    color: #374151;
}

.code-sample {
    background: #f8fafc;
    border: 1px solid #e5e7eb;
    border-radius: 0.5rem;
    padding: 1rem;
    font-family: 'Monaco', 'Menlo', 'Ubuntu Mono', monospace;
    font-size: 0.75rem;
    line-height: 1.4;
    color: #374151;
    margin-bottom: 1rem;
    overflow-x: auto;
}

.code-sample code {
    background: transparent;
    color: inherit;
}

/* Responsive Design */
@media (max-width: 1024px) {
    .admin-sidebar {
        transform: translateX(-100%);
        transition: transform 0.3s ease;
    }

    .admin-main {
        margin-left: 0;
    }

    .import-layout {
        grid-template-columns: 1fr;
        gap: 1.5rem;
    }

    .import-sidebar {
        position: static;
    }

    .form-card-body {
        padding: 1.5rem;
    }

    .info-card-body,
    .sample-card-body {
        padding: 1.5rem;
    }
}

@media (max-width: 768px) {
    .admin-header {
        padding: 1.5rem;
    }

    .admin-header h1 {
        font-size: 1.5rem;
    }

    .admin-content {
        padding: 1rem;
    }

    .form-actions {
        flex-direction: column;
    }

    .btn {
        justify-content: center;
    }
}
</style>