# Mutalex Academy LMS - Multi-Role Authentication System

## Overview

This document describes the comprehensive multi-role authentication system implemented for the Mutalex Academy Learning Management System (LMS). The system supports four user roles: Admin, Instructor, Student, and Guest, with proper access controls and security measures.

## Features

### User Roles
- **Admin**: Full system access, user management, content management, system settings
- **Instructor**: Course creation, student management, grading, content creation
- **Student**: Course enrollment, assignment submission, progress tracking
- **Guest**: Limited access to public content only

### Authentication Features
- Secure user registration with role selection
- Email verification for new accounts
- Password reset functionality
- Session management with automatic timeout
- Account lockout after failed login attempts
- CSRF protection
- Rate limiting
- Security headers and XSS protection

### Security Measures
- Password hashing with bcrypt
- Session security with regeneration
- SQL injection prevention
- XSS protection
- CSRF token validation
- File upload security
- Rate limiting and DDoS protection
- Security event logging

## Database Schema

### Users Table
```sql
CREATE TABLE users (
    id INT AUTO_INCREMENT PRIMARY KEY,
    username VARCHAR(50) UNIQUE NOT NULL,
    email VARCHAR(100) UNIQUE NOT NULL,
    password VARCHAR(255) NOT NULL,
    role ENUM('admin', 'instructor', 'student', 'guest') DEFAULT 'student',
    email_verified BOOLEAN DEFAULT FALSE,
    email_verification_token VARCHAR(255) NULL,
    password_reset_token VARCHAR(255) NULL,
    password_reset_expires TIMESTAMP NULL,
    last_login TIMESTAMP NULL,
    failed_login_attempts INT DEFAULT 0,
    account_locked_until TIMESTAMP NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);
```

## File Structure

```
c:/xampp/htdocs/lms/
├── config/
│   ├── auth.php          # Authentication functions
│   ├── rbac.php          # Role-Based Access Control
│   ├── security.php      # Security utilities
│   └── database.php      # Database connection
├── includes/
│   ├── middleware.php    # Authentication middleware
│   └── functions.php     # Utility functions
├── admin/
│   └── dashboard.php     # Admin dashboard
├── instructor/
│   └── dashboard.php     # Instructor dashboard
├── student/
│   └── dashboard.php     # Student dashboard
├── login.php             # Login page
├── register.php          # Registration page
├── forgot-password.php   # Password reset request
├── reset-password.php    # Password reset form
├── verify-email.php      # Email verification
├── logout.php            # Logout handler
├── logs/                 # Security logs
└── uploads/              # Secure file uploads
```

## API Reference

### Authentication Functions

#### Core Authentication
- `isLoggedIn()`: Check if user is authenticated
- `login($username, $password)`: Authenticate user
- `logout()`: Destroy session and logout
- `registerUser($username, $email, $password, $role)`: Register new user

#### Session Management
- `checkSessionTimeout()`: Check and handle session expiration
- `requireLogin()`: Require authentication for protected pages
- `getCurrentUser()`: Get current user information

#### Password Management
- `generatePasswordResetToken($email)`: Generate password reset token
- `resetPassword($token, $newPassword)`: Reset password using token
- `validatePassword($password)`: Validate password strength

#### Email Verification
- `verifyEmail($token)`: Verify email using token

### Role-Based Access Control (RBAC)

#### Role Checking
- `hasRole($role)`: Check if user has specific role
- `hasAnyRole($roles)`: Check if user has any of specified roles
- `hasPermission($permission)`: Check if user has specific permission

#### Permission Requirements
- `requireRole($role)`: Require specific role or redirect
- `requireAnyRole($roles)`: Require any of specified roles
- `requirePermission($permission)`: Require specific permission

#### Role Hierarchy
- `getRoleLevel($role)`: Get numerical role level
- `hasRoleLevel($minLevel)`: Check if user meets minimum role level

### Middleware Functions

#### Authentication Middleware
- `authMiddleware()`: Require authentication
- `roleMiddleware($role)`: Require specific role
- `anyRoleMiddleware($roles)`: Require any of roles
- `permissionMiddleware($permission)`: Require permission

#### Security Middleware
- `csrfMiddleware()`: Validate CSRF tokens
- `rateLimitMiddleware()`: Apply rate limiting
- `securityHeaders()`: Set security headers

### Security Utilities

#### CSRF Protection
- `generateCSRFToken()`: Generate CSRF token
- `validateCSRFToken($token)`: Validate CSRF token

#### Input Security
- `sanitizeInput($input)`: Sanitize user input
- `secureOutput($data)`: Secure output for display

#### File Security
- `handleFileUpload($file)`: Secure file upload
- `validatePassword($password)`: Password validation

## Usage Examples

### Protecting a Page
```php
<?php
require_once '../config/auth.php';
require_once '../config/rbac.php';
require_once '../includes/middleware.php';

// Require admin role
adminOnly();

// Or require any role
anyRoleMiddleware(['admin', 'instructor']);

// Or check permission
if (!hasPermission('manage_users')) {
    header('Location: ../access-denied.php');
    exit;
}
?>
```

### User Registration
```php
$result = registerUser('john_doe', 'john@example.com', 'password123', 'student');
if ($result['success']) {
    // Send verification email
    // Redirect to success page
} else {
    // Handle error
    echo $result['message'];
}
```

### Login Process
```php
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $username = sanitizeInput($_POST['username']);
    $password = $_POST['password'];

    if (login($username, $password)) {
        header('Location: ' . getDashboardUrl());
        exit;
    } else {
        $error = 'Invalid credentials';
    }
}
```

### Checking Permissions in Templates
```php
<?php if (hasPermission('manage_users')): ?>
    <a href="users.php">Manage Users</a>
<?php endif; ?>

<?php if (hasRole('admin')): ?>
    <a href="settings.php">System Settings</a>
<?php endif; ?>
```

## Security Best Practices

### Password Policy
- Minimum 8 characters
- Must contain uppercase and lowercase letters
- Must contain numbers
- Optional: special characters

### Session Security
- 30-minute session timeout
- Session ID regeneration every 10 minutes
- Secure session cookies
- Session fixation protection

### Account Security
- Account lockout after 5 failed attempts
- 15-minute lockout duration
- Failed attempt tracking
- Last login tracking

### Network Security
- HTTPS enforcement in production
- Security headers (CSP, HSTS, X-Frame-Options)
- CSRF protection on all forms
- Rate limiting on sensitive endpoints

### Database Security
- Prepared statements for all queries
- Input sanitization
- Password hashing with bcrypt
- SQL injection prevention

## Configuration

### Security Settings (config/security.php)
```php
define('PASSWORD_MIN_LENGTH', 8);
define('SESSION_LIFETIME', 30 * 60);
define('MAX_LOGIN_ATTEMPTS', 5);
define('LOGIN_LOCKOUT_TIME', 15 * 60);
```

### Role Permissions (config/rbac.php)
```php
define('PERMISSIONS', [
    'view_posts' => ['guest', 'student', 'instructor', 'admin'],
    'create_posts' => ['instructor', 'admin'],
    'manage_users' => ['admin'],
    // ... more permissions
]);
```

## Testing

### Default Test Accounts
- **Admin**: username: `admin`, password: `admin123`
- **Instructor**: Create via registration or database
- **Student**: Create via registration or database

### Testing Scenarios
1. Login with different roles and verify redirects
2. Test password reset functionality
3. Test account lockout after failed attempts
4. Test session timeout
5. Test CSRF protection
6. Test role-based access to different pages

## Future Enhancements

### Planned Features
- Two-factor authentication (2FA)
- OAuth integration (Google, Facebook)
- Advanced user management
- Audit logging
- API authentication tokens
- Remember me functionality
- Social login integration

### Integration Points
- Email service integration (SendGrid, Mailgun)
- SMS verification for 2FA
- Redis for session storage and caching
- Advanced rate limiting with Redis
- User activity monitoring

## Troubleshooting

### Common Issues
1. **Session not persisting**: Check PHP session configuration
2. **CSRF token errors**: Ensure tokens are generated and validated
3. **Role access denied**: Verify user roles in database
4. **Email not sending**: Configure SMTP settings

### Debug Mode
Enable debug logging in config files:
```php
define('DEBUG_MODE', true);
error_reporting(E_ALL);
ini_set('display_errors', 1);
```

## Support

For technical support or questions about the authentication system:
- Check the security logs in `logs/security.log`
- Review PHP error logs
- Verify database connectivity
- Test with default admin account

## License

This authentication system is part of the Mutalex Academy LMS project.