# PayNow Payment Integration

This document describes the official PayNow API integration for the LMS platform.

## Overview

This integration uses the official PayNow REST API for Zimbabwe. Documentation: https://developers.paynow.co.zw/

## Features

- ✅ Payment session creation
- ✅ Payment status checking via poll URL
- ✅ Webhook handling for automatic status updates
- ✅ Signature verification for security
- ✅ Proper error handling
- ✅ Sandbox/Live environment support
- ✅ Mobile number formatting (Zimbabwe format)

## Files

```
includes/paynow/
├── PayNowClient.php          # Official API client implementation
├── paynow_integration.php    # Helper functions for easy integration
└── index.php                 # Autoloader

api/webhooks/
└── paynow.php                # Webhook endpoint for payment notifications

add_paynow_fields_migration.php  # Database migration script
test_paynow_integration.php      # Test suite
```

## Configuration

### 1. Get PayNow Credentials

1. Go to https://www.paynow.co.zw/developer/integration
2. Create a developer account
3. Get your Integration ID and Integration Key
4. For testing, use sandbox credentials

### 2. Database Configuration

Run the migration to add required fields:

```bash
php add_paynow_fields_migration.php
```

### 3. Update Payment Gateway Settings

Insert or update PayNow settings in `payment_gateway_settings` table:

```sql
INSERT INTO payment_gateway_settings
(gateway_name, gateway_type, is_active, api_key, api_secret, 
 merchant_id, merchant_email, merchant_mobile, webhook_url, return_url)
VALUES 
('paynow', 'sandbox', 1, 'YOUR_INTEGRATION_ID', 'YOUR_INTEGRATION_KEY',
 'YOUR_MERCHANT_ID', 'payments@yoursite.com', '263771234567',
 'https://yoursite.com/api/webhooks/paynow.php',
 'https://yoursite.com/student/checkout.php?status=callback');
```

### 4. Environment Variables (.env)

```env
# PayNow Configuration
PAYNOW_INTEGRATION_ID=your_integration_id
PAYNOW_INTEGRATION_KEY=your_integration_key
PAYNOW_MERCHANT_ID=your_merchant_id
```

## API Usage

### Basic Payment Creation

```php
use PayNow\PayNowClient;
use PayNow\paynow_integration;

// Using helper function
$result = createPayNowPayment(
    userId: 123,
    itemType: 'course',
    itemId: 456,
    amount: 99.99,
    itemName: 'Web Development Course',
    discountCode: null,
    environment: 'sandbox'
);

if ($result['success']) {
    // Redirect user to PayNow payment page
    header('Location: ' . $result['redirect_url']);
    exit;
}
```

### Using Client Directly

```php
$client = new PayNowClient(
    integrationId: 'YOUR_INTEGRATION_ID',
    integrationKey: 'YOUR_INTEGRATION_KEY',
    merchantId: 'YOUR_MERCHANT_ID',
    isStaging: true,
    merchantEmail: 'payments@yoursite.com',
    merchantMobile: '263771234567'
);

$response = $client->createPayment(
    reference: 'ORDER_123',
    amount: 99.99,
    description: 'Course Purchase',
    buyerEmail: 'customer@email.com',
    buyerMobile: '0771234567'
);

if ($response['success']) {
    // Redirect to payment page
    header('Location: ' . $response['redirect_url']);
}
```

### Check Payment Status

```php
$result = checkPayNowPaymentStatus('transaction_id', 'sandbox');

if ($result['success'] && $result['paid']) {
    // Payment successful - process enrollment
}
```

### Handle Webhook Callback

```php
// In api/webhooks/paynow.php
$data = json_decode(file_get_contents('php://input'), true);

// Webhook is automatically processed by the endpoint
// Status updates are handled and enrollment is processed
```

## Payment Flow

```
1. User selects PayNow at checkout
2. Create payment session → Get redirect URL
3. Redirect user to PayNow payment page
4. User completes payment
5. PayNow sends webhook → Update status in database
6. User returns to site → Check status → Show confirmation
```

## Webhook Endpoints

| Environment | URL |
|-------------|-----|
| Sandbox | `https://yoursite.com/api/webhooks/paynow.php` |
| Live | `https://yoursite.com/api/webhooks/paynow.php` |

## Status Mapping

| PayNow Status | Internal Status | Action |
|---------------|-----------------|--------|
| Paid | completed | Process enrollment |
| Pending | processing | Wait for webhook |
| Cancelled | cancelled | Notify user |
| Failed | failed | Notify user |
| Created | pending | Initial state |

## Testing

Run the test suite:

```bash
php test_paynow_integration.php
```

### Test Credentials

Use PayNow's sandbox credentials from: https://developers.paynow.co.zw/

## Error Handling

All errors are logged to `payment_logs` table:

```php
// Check for errors
if (!$result['success']) {
    echo 'Error: ' . $result['error'];
    // Log is automatically created
}
```

## Security

- ✅ SHA256 hash verification for all requests
- ✅ Webhook signature verification
- ✅ CSRF protection for payment forms
- ✅ Rate limiting on payment endpoints
- ✅ HTTPS required for production

## Deployment Checklist

- [ ] Get production credentials from PayNow
- [ ] Update database with production settings
- [ ] Change `gateway_type` from 'sandbox' to 'live'
- [ ] Test complete payment flow
- [ ] Verify webhook endpoint is accessible
- [ ] Check SSL certificate is valid
- [ ] Monitor payment logs for errors

## Troubleshooting

### Payment not initiated
- Check API credentials in database
- Verify Integration ID and Key are correct
- Check payment_logs for error details

### Webhook not received
- Verify webhook URL is publicly accessible
- Check server logs for incoming requests
- Ensure firewall allows incoming requests

### Hash verification failed
- Ensure integration key matches exactly
- Check hash generation order
- Verify hash is uppercase SHA256

## Support

- PayNow Developer Portal: https://developers.paynow.co.zw/
- PayNow Support: support@paynow.co.zw
