<?php
require_once 'BaseAIService.php';

class AccessibilityService extends BaseAIService {

    protected function isFeatureEnabled() {
        return isAIFeatureEnabled('accessibility');
    }

    protected function getFeatureName() {
        return 'accessibility';
    }

    protected function validateInput($data) {
        if (!isset($data['content']) || empty($data['content'])) {
            throw new Exception('Content is required for accessibility analysis');
        }
        return true;
    }

    public function analyzeAccessibility($content, $standards = ['WCAG_2_1']) {
        $this->validateInput(['content' => $content]);

        $prompt = $this->buildAccessibilityPrompt($content, $standards);

        $messages = [
            [
                'role' => 'system',
                'content' => 'You are an accessibility expert. Analyze content for compliance with accessibility standards and provide improvement suggestions.'
            ],
            [
                'role' => 'user',
                'content' => $prompt
            ]
        ];

        $response = $this->makeRequest($messages);

        return $this->parseAccessibilityResponse($response->choices[0]->message->content);
    }

    private function buildAccessibilityPrompt($content, $standards) {
        $standardsText = implode(', ', $standards);
        $prompt = "Analyze the following content for accessibility compliance with {$standardsText}.\n\n";
        $prompt .= "Content:\n{$content}\n\n";
        $prompt .= "Check for:\n";
        $prompt .= "- Alternative text for images\n";
        $prompt .= "- Color contrast issues\n";
        $prompt .= "- Semantic HTML structure\n";
        $prompt .= "- Keyboard navigation\n";
        $prompt .= "- Screen reader compatibility\n";
        $prompt .= "- Readability and language complexity\n\n";
        $prompt .= "Provide specific recommendations for improvement.";

        return $prompt;
    }

    private function parseAccessibilityResponse($response) {
        return [
            'compliance_score' => $this->extractScore($response),
            'issues' => $this->extractIssues($response),
            'recommendations' => $this->extractRecommendations($response),
            'priority_fixes' => $this->extractPriorityFixes($response)
        ];
    }

    private function extractScore($response) {
        if (preg_match('/score[^0-9]*([0-9]+(?:\.[0-9]+)?)/i', $response, $match)) {
            return (float) $match[1];
        }
        return 0;
    }

    private function extractIssues($response) {
        return $this->extractList($response, 'issues|problems');
    }

    private function extractRecommendations($response) {
        return $this->extractList($response, 'recommendations|suggestions');
    }

    private function extractPriorityFixes($response) {
        return $this->extractList($response, 'priority|critical');
    }

    private function extractList($response, $keywords) {
        $items = [];
        if (preg_match_all('/(?:^|\n)[•\-\*]\s*(.*?)(?=\n[•\-\*]|\n\n|$)/', $response, $matches)) {
            $items = $matches[1];
        }
        return $items;
    }
}
?>