<?php
require_once 'BaseAIService.php';

class LearningPathService extends BaseAIService {

    protected function isFeatureEnabled() {
        return isAIFeatureEnabled('learning_path');
    }

    protected function getFeatureName() {
        return 'learning_path';
    }

    protected function validateInput($data) {
        if (!isset($data['course_content']) || empty($data['course_content'])) {
            throw new Exception('Course content is required');
        }
        return true;
    }

    public function generateLearningPath($courseContent, $studentLevel = 'intermediate') {
        $this->validateInput(['course_content' => $courseContent]);

        $prompt = $this->buildPathPrompt($courseContent, $studentLevel);

        $messages = [
            [
                'role' => 'system',
                'content' => 'You are an expert curriculum designer. Create optimal learning sequences and identify prerequisites.'
            ],
            [
                'role' => 'user',
                'content' => $prompt
            ]
        ];

        $response = $this->makeRequest($messages);

        return $this->parsePathResponse($response->choices[0]->message->content);
    }

    private function buildPathPrompt($content, $level) {
        $prompt = "Analyze the following course content and create an optimal learning path.\n\n";
        $prompt .= "Student Level: {$level}\n\n";
        $prompt .= "Content:\n{$content}\n\n";
        $prompt .= "Provide:\n1. Prerequisite knowledge required\n2. Recommended learning sequence\n3. Key concepts to master at each stage\n4. Estimated time for each section\n5. Assessment points";

        return $prompt;
    }

    private function parsePathResponse($response) {
        return [
            'prerequisites' => $this->extractSection($response, 'prerequisites'),
            'sequence' => $this->extractSequence($response),
            'key_concepts' => $this->extractKeyConcepts($response),
            'time_estimates' => $this->extractTimeEstimates($response),
            'assessments' => $this->extractAssessments($response)
        ];
    }

    private function extractSection($response, $section) {
        if (preg_match("/{$section}[^:]*:?\s*(.*?)(?=\n[A-Z]|\n\s*\d+\.|$)/is", $response, $match)) {
            return trim($match[1]);
        }
        return '';
    }

    private function extractSequence($response) {
        $sequence = [];
        if (preg_match_all('/(?:step|module|section)\s*\d*[^:]*:?\s*(.*?)(?=\n(?:step|module|section)|\n\n|$)/is', $response, $matches)) {
            $sequence = array_map('trim', $matches[1]);
        }
        return $sequence;
    }

    private function extractKeyConcepts($response) {
        return $this->extractList($response, 'concepts');
    }

    private function extractTimeEstimates($response) {
        $times = [];
        if (preg_match_all('/(\d+)\s*(?:hours?|hrs?)/i', $response, $matches)) {
            $times = $matches[1];
        }
        return $times;
    }

    private function extractAssessments($response) {
        return $this->extractList($response, 'assessments');
    }

    private function extractList($response, $keyword) {
        $items = [];
        if (preg_match_all('/(?:^|\n)[•\-\*]\s*(.*?)(?=\n[•\-\*]|\n\n|$)/', $response, $matches)) {
            $items = $matches[1];
        }
        return $items;
    }
}
?>