<?php
require_once '../../config/database.php';
require_once '../../config/auth.php';
require_once '../../config/rbac.php';
$lms_config = require '../../config/lms.php';
require_once '../../includes/functions.php';
require_once '../../includes/middleware.php';

// Check if user is instructor or admin
anyRoleMiddleware(['instructor', 'admin']);

$user = getUserContext();
$page_title = 'Question Analysis - ' . $lms_config['lms']['site_name'];
?>

<div class="admin-container">
    <div class="admin-sidebar">
        <div class="sidebar-header">
            <h2><i class="fas fa-graduation-cap me-2"></i><?php echo htmlspecialchars($lms_config['lms']['site_name']); ?></h2>
            <p><?php echo htmlspecialchars($lms_config['lms']['site_name']); ?></p>
        </div>

        <nav class="sidebar-nav">
            <ul>
                <li><a href="../dashboard.php"><i class="fas fa-tachometer-alt me-2"></i>Dashboard</a></li>
                <li><a href="../courses/"><i class="fas fa-book me-2"></i>My Courses</a></li>
                <li><a href="../courses/new.php"><i class="fas fa-plus me-2"></i>Create Course</a></li>
                <li><a href="index.php" class="active"><i class="fas fa-question-circle me-2"></i>Question Bank</a></li>
                <li><a href="../exams/"><i class="fas fa-file-alt me-2"></i>Exams</a></li>
                <li><a href="../resources.php"><i class="fas fa-folder me-2"></i>Resources</a></li>
                <li><a href="../analytics.php"><i class="fas fa-chart-line me-2"></i>Analytics</a></li>
                <li><a href="../../logout.php"><i class="fas fa-sign-out-alt me-2"></i>Logout</a></li>
            </ul>
        </nav>
    </div>

    <div class="admin-main">
        <div class="admin-header">
            <div class="d-flex justify-content-between align-items-center">
                <div>
                    <h1><i class="fas fa-chart-bar text-info me-3"></i>Question Item Analysis</h1>
                    <p class="text-muted mb-0">Analyze question performance and identify areas for improvement</p>
                </div>
                <div class="header-actions">
                    <a href="index.php" class="btn btn-outline-primary btn-lg">
                        <i class="fas fa-arrow-left me-2"></i>Back to Questions
                    </a>
                    <button id="export-analysis" class="btn btn-success btn-lg">
                        <i class="fas fa-download me-2"></i>Export Analysis
                    </button>
                </div>
            </div>
        </div>

        <div class="admin-content">

            <!-- Filters -->
            <div class="filter-section">
                <div class="filter-form">
                    <div class="filter-row">
                        <div class="filter-group">
                            <label for="course-filter"><i class="fas fa-book me-1"></i>Course</label>
                            <select id="course-filter" class="form-select">
                                <option value="">All Courses</option>
                                <!-- Courses will be populated by JavaScript -->
                            </select>
                        </div>
                        <div class="filter-group">
                            <label for="difficulty-filter"><i class="fas fa-chart-line me-1"></i>Difficulty</label>
                            <select id="difficulty-filter" class="form-select">
                                <option value="">All Levels</option>
                                <option value="easy">Easy</option>
                                <option value="medium">Medium</option>
                                <option value="hard">Hard</option>
                            </select>
                        </div>
                        <div class="filter-group">
                            <label for="type-filter"><i class="fas fa-cogs me-1"></i>Question Type</label>
                            <select id="type-filter" class="form-select">
                                <option value="">All Types</option>
                                <option value="mcq_single">Multiple Choice (Single)</option>
                                <option value="mcq_multiple">Multiple Choice (Multiple)</option>
                                <option value="true_false">True/False</option>
                                <option value="essay">Essay</option>
                                <option value="short_answer">Short Answer</option>
                            </select>
                        </div>
                        <div class="filter-group">
                            <label for="performance-filter"><i class="fas fa-trophy me-1"></i>Performance</label>
                            <select id="performance-filter" class="form-select">
                                <option value="">All Questions</option>
                                <option value="poor">Poor (< 30%)</option>
                                <option value="fair">Fair (30-70%)</option>
                                <option value="good">Good (> 70%)</option>
                            </select>
                        </div>
                        <div class="filter-actions">
                            <button id="apply-filters" class="btn btn-primary">
                                <i class="fas fa-filter me-1"></i>Apply Filters
                            </button>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Analysis Summary -->
            <div class="analysis-summary" id="analysis-summary">
                <div class="summary-card">
                    <div class="stat-icon">
                        <i class="fas fa-question-circle"></i>
                    </div>
                    <div class="stat-content">
                        <h3 id="total-questions">-</h3>
                        <p>Total Questions</p>
                    </div>
                </div>
                <div class="summary-card">
                    <div class="stat-icon">
                        <i class="fas fa-chart-line"></i>
                    </div>
                    <div class="stat-content">
                        <h3 id="avg-success-rate">0%</h3>
                        <p>Average Success Rate</p>
                    </div>
                </div>
                <div class="summary-card">
                    <div class="stat-icon">
                        <i class="fas fa-exclamation-triangle"></i>
                    </div>
                    <div class="stat-content">
                        <h3 id="difficult-questions">-</h3>
                        <p>Difficult Questions</p>
                    </div>
                </div>
                <div class="summary-card">
                    <div class="stat-icon">
                        <i class="fas fa-check-circle"></i>
                    </div>
                    <div class="stat-content">
                        <h3 id="easy-questions">-</h3>
                        <p>Easy Questions</p>
                    </div>
                </div>
            </div>

            <!-- Questions Analysis Table -->
            <div class="questions-analysis-container">
                <div class="table-header">
                    <h3><i class="fas fa-table me-2"></i>Question Performance Analysis</h3>
                </div>
                <div class="questions-analysis-table" id="questions-table">
                    <div class="loading">
                        <div class="loading-spinner"></div>
                        <p>Loading question analysis...</p>
                    </div>
                </div>
            </div>

        <!-- Detailed Analysis Modal -->
        <div id="question-detail-modal" class="modal fade" tabindex="-1">
            <div class="modal-dialog modal-xl">
                <div class="modal-content">
                    <div class="modal-header">
                        <h5 class="modal-title">Question Details</h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                    </div>
                    <div class="modal-body" id="question-detail-content">
                        <!-- Content will be populated here -->
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
let currentFilters = {};

document.addEventListener('DOMContentLoaded', function() {
    loadCourses();
    loadQuestionAnalysis();

    // Event listeners
    document.getElementById('apply-filters').addEventListener('click', function() {
        applyFilters();
    });

    document.getElementById('export-analysis').addEventListener('click', exportAnalysis);

    // Enter key for filters
    ['course-filter', 'difficulty-filter', 'type-filter', 'performance-filter'].forEach(id => {
        document.getElementById(id).addEventListener('change', function() {
            // Auto-apply filters on change
            setTimeout(applyFilters, 300);
        });
    });
});

function loadCourses() {
    fetch('../../api/analytics/index.php?action=course_analytics&get_courses=1')
        .then(response => response.json())
        .then(data => {
            // This would populate course filter - simplified for now
            console.log('Courses loaded');
        })
        .catch(error => console.error('Error loading courses:', error));
}

function applyFilters() {
    currentFilters = {
        course_id: document.getElementById('course-filter').value,
        difficulty: document.getElementById('difficulty-filter').value,
        type: document.getElementById('type-filter').value,
        performance: document.getElementById('performance-filter').value
    };

    loadQuestionAnalysis();
}

function loadQuestionAnalysis() {
    const params = new URLSearchParams({
        action: 'question_analysis',
        ...currentFilters
    });

    fetch(`../../api/analytics/index.php?${params}`)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                updateAnalysisSummary(data.data);
                updateQuestionsTable(data.data);
            } else {
                showError('Failed to load question analysis');
            }
        })
        .catch(error => {
            console.error('Error loading question analysis:', error);
            showError('Error loading question analysis');
        });
}

function updateAnalysisSummary(data) {
    // Calculate summary statistics
    const questions = data.questions || [];
    const totalQuestions = questions.length;
    const avgSuccessRate = questions.length > 0
        ? (questions.reduce((sum, q) => sum + (q.success_rate || 0), 0) / questions.length).toFixed(1)
        : 0;
    const difficultQuestions = questions.filter(q => (q.success_rate || 0) < 30).length;
    const easyQuestions = questions.filter(q => (q.success_rate || 0) > 80).length;

    document.getElementById('total-questions').textContent = totalQuestions;
    document.getElementById('avg-success-rate').textContent = avgSuccessRate + '%';
    document.getElementById('difficult-questions').textContent = difficultQuestions;
    document.getElementById('easy-questions').textContent = easyQuestions;
}

function updateQuestionsTable(data) {
    const container = document.getElementById('questions-table');
    const questions = data.questions || [];

    if (questions.length === 0) {
        container.innerHTML = '<p class="text-muted text-center py-4">No questions found matching the criteria.</p>';
        return;
    }

    container.innerHTML = `
        <table class="table table-striped">
            <thead>
                <tr>
                    <th>Question</th>
                    <th>Type</th>
                    <th>Difficulty</th>
                    <th>Attempts</th>
                    <th>Success Rate</th>
                    <th>Avg. Time</th>
                    <th>Discrimination</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
                ${questions.map(question => `
                    <tr>
                        <td>
                            <div class="question-preview">
                                ${question.title.length > 50 ? question.title.substring(0, 50) + '...' : question.title}
                            </div>
                        </td>
                        <td>
                            <span class="badge bg-secondary">${formatQuestionType(question.question_type)}</span>
                        </td>
                        <td>
                            <span class="badge bg-${getDifficultyColor(question.difficulty || 'medium')}">
                                ${question.difficulty || 'medium'}
                            </span>
                        </td>
                        <td>${question.total_attempts || 0}</td>
                        <td>
                            <div class="success-rate">
                                <div class="progress-bar">
                                    <div class="progress-fill" style="width: ${question.success_rate || 0}%"></div>
                                </div>
                                <span class="rate-text">${(question.success_rate || 0).toFixed(1)}%</span>
                            </div>
                        </td>
                        <td>${question.average_time ? Math.round(question.average_time / 60) + 'm' : 'N/A'}</td>
                        <td>
                            <span class="discrimination ${getDiscriminationClass(question.success_rate || 0)}">
                                ${getDiscriminationLabel(question.success_rate || 0)}
                            </span>
                        </td>
                        <td>
                            <button class="btn btn-sm btn-outline-primary" onclick="viewQuestionDetails(${question.id})">
                                Details
                            </button>
                        </td>
                    </tr>
                `).join('')}
            </tbody>
        </table>
    `;
}

function formatQuestionType(type) {
    const types = {
        'mcq_single': 'MCQ Single',
        'mcq_multiple': 'MCQ Multiple',
        'true_false': 'True/False',
        'essay': 'Essay',
        'short_answer': 'Short Answer',
        'matching': 'Matching',
        'code': 'Code',
        'fill_blanks': 'Fill Blanks'
    };
    return types[type] || type;
}

function getDifficultyColor(difficulty) {
    switch (difficulty.toLowerCase()) {
        case 'easy': return 'success';
        case 'medium': return 'warning';
        case 'hard': return 'danger';
        default: return 'secondary';
    }
}

function getDiscriminationClass(rate) {
    if (rate >= 80) return 'excellent';
    if (rate >= 60) return 'good';
    if (rate >= 40) return 'fair';
    return 'poor';
}

function getDiscriminationLabel(rate) {
    if (rate >= 80) return 'Excellent';
    if (rate >= 60) return 'Good';
    if (rate >= 40) return 'Fair';
    return 'Poor';
}

function viewQuestionDetails(questionId) {
    const modal = new bootstrap.Modal(document.getElementById('question-detail-modal'));
    const content = document.getElementById('question-detail-content');

    // Fetch detailed question analysis
    fetch(`../../api/analytics/index.php?action=question_analysis&question_id=${questionId}`)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                content.innerHTML = generateQuestionDetailHTML(data.data);
            } else {
                content.innerHTML = '<p class="text-danger">Failed to load question details.</p>';
            }
            modal.show();
        })
        .catch(error => {
            content.innerHTML = '<p class="text-danger">Error loading question details.</p>';
            modal.show();
        });
}

function generateQuestionDetailHTML(data) {
    const question = data.question;
    const stats = data.statistics;
    const optionAnalysis = data.option_analysis || [];
    const difficulty = data.difficulty_analysis;

    return `
        <div class="question-detail">
            <div class="question-content">
                <h6>Question:</h6>
                <div class="question-text">${question.title}</div>
                <div class="question-meta">
                    <span class="badge bg-secondary">${formatQuestionType(question.question_type)}</span>
                    <span class="badge bg-${getDifficultyColor(question.difficulty)}">${question.difficulty}</span>
                    <span class="badge bg-info">${question.points} points</span>
                </div>
            </div>

            <div class="analysis-metrics">
                <div class="metric-grid">
                    <div class="metric-item">
                        <h4>${stats.total_attempts}</h4>
                        <p>Total Attempts</p>
                    </div>
                    <div class="metric-item">
                        <h4>${stats.success_rate}%</h4>
                        <p>Success Rate</p>
                    </div>
                    <div class="metric-item">
                        <h4>${stats.avg_time ? Math.round(stats.avg_time / 60) + 'm' : 'N/A'}</h4>
                        <p>Average Time</p>
                    </div>
                    <div class="metric-item">
                        <h4>${difficulty.calculated_difficulty}</h4>
                        <p>Calculated Difficulty</p>
                        </div>
                    </div>
                </div>
            </div>

            ${optionAnalysis.length > 0 ? `
                <div class="option-analysis">
                    <h6>Option Analysis:</h6>
                    <table class="table table-sm">
                        <thead>
                            <tr>
                                <th>Option</th>
                                <th>Times Selected</th>
                                <th>Selection Rate</th>
                                <th>Correct</th>
                            </tr>
                        </thead>
                        <tbody>
                            ${optionAnalysis.map(option => `
                                <tr>
                                    <td>${option.option_text.substring(0, 50)}${option.option_text.length > 50 ? '...' : ''}</td>
                                    <td>${option.times_selected}</td>
                                    <td>${stats.total_attempts > 0 ? ((option.times_selected / stats.total_attempts) * 100).toFixed(1) : 0}%</td>
                                    <td>${option.is_correct ? '<i class="fas fa-check text-success"></i>' : '<i class="fas fa-times text-danger"></i>'}</td>
                                </tr>
                            `).join('')}
                        </tbody>
                    </table>
                </div>
            ` : ''}

            <div class="recommendations">
                <h6>Recommendations:</h6>
                <ul>
                    ${difficulty.recommendations.map(rec => `<li>${rec}</li>`).join('')}
                </ul>
            </div>
        </div>
    `;
}

function exportAnalysis() {
    const params = new URLSearchParams({
        action: 'export_data',
        type: 'question_analysis',
        format: 'csv',
        ...currentFilters
    });

    const link = document.createElement('a');
    link.href = `../../api/analytics/index.php?${params}`;
    link.download = 'question_analysis_export.csv';
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
}

function showError(message) {
    const errorDiv = document.createElement('div');
    errorDiv.className = 'alert alert-danger';
    errorDiv.textContent = message;
    document.querySelector('.admin-content').prepend(errorDiv);

    setTimeout(() => {
        errorDiv.remove();
    }, 5000);
}
</script>

<style>
/* Custom styles for Question Analysis page */
.admin-container {
    display: flex;
    min-height: 100vh;
    background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
}

.admin-sidebar {
    width: 280px;
    background: linear-gradient(135deg, #1e293b 0%, #334155 100%);
    color: white;
    position: fixed;
    height: 100vh;
    overflow-y: auto;
    box-shadow: 4px 0 15px rgba(0, 0, 0, 0.1);
    z-index: 1000;
}

.sidebar-header {
    padding: 2rem 1.5rem;
    border-bottom: 1px solid #374151;
    text-align: center;
}

.sidebar-header h2 {
    font-size: 1.5rem;
    font-weight: 800;
    margin-bottom: 0.5rem;
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 0.5rem;
}

.sidebar-nav ul {
    list-style: none;
    padding: 0;
    margin: 0;
}

.sidebar-nav li a {
    display: flex;
    align-items: center;
    gap: 0.75rem;
    padding: 1rem 1.5rem;
    color: #cbd5e1;
    text-decoration: none;
    transition: all 0.3s ease;
    font-weight: 500;
    border-left: 3px solid transparent;
}

.sidebar-nav li a:hover,
.sidebar-nav li a.active {
    background: linear-gradient(135deg, #374151 0%, #4b5563 100%);
    color: white;
    border-left-color: #3b82f6;
    transform: translateX(5px);
}

.admin-main {
    margin-left: 280px;
    flex: 1;
    min-height: 100vh;
}

.admin-header {
    background: white;
    padding: 2rem;
    border-bottom: 1px solid #e5e7eb;
    box-shadow: 0 2px 10px rgba(0, 0, 0, 0.05);
}

.admin-header h1 {
    font-size: 2rem;
    font-weight: 800;
    color: #1f2937;
    margin-bottom: 0.5rem;
    display: flex;
    align-items: center;
    gap: 0.75rem;
}

.header-actions {
    display: flex;
    gap: 1rem;
    align-items: center;
}

.admin-content {
    padding: 2rem;
}

.filter-section {
    background: white;
    border-radius: 1rem;
    padding: 2rem;
    margin-bottom: 2rem;
    box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
    border: 1px solid rgba(255, 255, 255, 0.8);
}

.filter-row {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 1.5rem;
    align-items: end;
}

.filter-group label {
    font-weight: 600;
    color: #374151;
    margin-bottom: 0.5rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.filter-actions {
    display: flex;
    gap: 0.75rem;
    align-items: end;
}

.analysis-summary {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 1.5rem;
    margin-bottom: 2rem;
}

.summary-card {
    background: white;
    border-radius: 1rem;
    padding: 1.5rem;
    box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
    border: 1px solid rgba(255, 255, 255, 0.8);
    transition: all 0.3s ease;
    display: flex;
    align-items: center;
    gap: 1rem;
}

.summary-card:hover {
    transform: translateY(-5px);
    box-shadow: 0 8px 25px rgba(0, 0, 0, 0.15);
}

.stat-icon {
    width: 60px;
    height: 60px;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    border-radius: 1rem;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.5rem;
    flex-shrink: 0;
    box-shadow: 0 4px 15px rgba(102, 126, 234, 0.3);
}

.stat-content h3 {
    font-size: 2rem;
    font-weight: 800;
    color: #1f2937;
    margin-bottom: 0.25rem;
    line-height: 1;
}

.stat-content p {
    color: #64748b;
    font-weight: 500;
    margin: 0;
    font-size: 0.9rem;
}

.questions-analysis-container {
    background: white;
    border-radius: 1rem;
    box-shadow: 0 8px 32px rgba(0, 0, 0, 0.1);
    overflow: hidden;
    border: 1px solid rgba(255, 255, 255, 0.8);
}

.table-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 2rem;
    background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
    border-bottom: 1px solid #e5e7eb;
}

.table-header h3 {
    font-size: 1.5rem;
    font-weight: 700;
    color: #1f2937;
    margin: 0;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.questions-analysis-table {
    background: white;
}

.table {
    width: 100%;
    margin-bottom: 0;
}

.table th,
.table td {
    padding: 1rem;
    vertical-align: middle;
    border-top: 1px solid #e5e7eb;
}

.table thead th {
    border-bottom: 2px solid #e5e7eb;
    background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
    font-weight: 700;
    color: #374151;
    font-size: 0.875rem;
    text-transform: uppercase;
    letter-spacing: 0.05em;
}

.question-preview {
    max-width: 300px;
    overflow: hidden;
    text-overflow: ellipsis;
    white-space: nowrap;
    font-weight: 600;
    color: #1f2937;
}

.success-rate {
    display: flex;
    align-items: center;
    gap: 10px;
}

.progress-bar {
    flex: 1;
    height: 8px;
    background: #e9ecef;
    border-radius: 4px;
    overflow: hidden;
}

.progress-fill {
    height: 100%;
    background: linear-gradient(135deg, #10b981 0%, #059669 100%);
    border-radius: 4px;
    transition: width 0.3s ease;
}

.rate-text {
    font-size: 12px;
    font-weight: bold;
    color: #1f2937;
    min-width: 35px;
}

.discrimination {
    font-weight: bold;
}

.discrimination.excellent { color: #10b981; }
.discrimination.good { color: #059669; }
.discrimination.fair { color: #f59e0b; }
.discrimination.poor { color: #dc2626; }

.loading {
    text-align: center;
    padding: 4rem 2rem;
    color: #64748b;
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 1rem;
}

.loading-spinner {
    width: 40px;
    height: 40px;
    border: 4px solid #e5e7eb;
    border-top: 4px solid #667eea;
    border-radius: 50%;
    animation: spin 1s linear infinite;
}

@keyframes spin {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
}

.question-detail {
    padding: 2rem;
}

.question-content {
    margin-bottom: 2rem;
    padding: 1.5rem;
    background: #f8fafc;
    border-radius: 1rem;
    border: 1px solid #e5e7eb;
}

.question-text {
    font-size: 1.1rem;
    margin: 1rem 0;
    color: #1f2937;
    line-height: 1.6;
}

.question-meta {
    display: flex;
    gap: 1rem;
    flex-wrap: wrap;
}

.analysis-metrics {
    margin-bottom: 2rem;
}

.metric-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 1rem;
}

.metric-item {
    text-align: center;
    padding: 1.5rem;
    background: white;
    border-radius: 1rem;
    box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
    border: 1px solid #e5e7eb;
}

.metric-item h4 {
    margin: 0 0 0.5rem 0;
    color: #667eea;
    font-size: 2rem;
    font-weight: 800;
}

.metric-item p {
    margin: 0;
    color: #64748b;
    font-weight: 500;
    font-size: 0.9rem;
}

.option-analysis,
.recommendations {
    margin-bottom: 2rem;
}

.option-analysis h6,
.recommendations h6 {
    margin-bottom: 1rem;
    color: #1f2937;
    font-weight: 600;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.recommendations ul {
    padding-left: 1.5rem;
}

.recommendations li {
    margin-bottom: 0.5rem;
    color: #374151;
    line-height: 1.5;
}

.alert {
    padding: 1rem 1.5rem;
    margin-bottom: 2rem;
    border: 1px solid transparent;
    border-radius: 0.75rem;
    box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
}

.alert-danger {
    color: #dc2626;
    background-color: #fef2f2;
    border-color: #fecaca;
}

.btn-primary {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    border: none;
    border-radius: 0.75rem;
    padding: 0.75rem 1.5rem;
    font-weight: 600;
    transition: all 0.3s ease;
    box-shadow: 0 4px 15px rgba(102, 126, 234, 0.3);
}

.btn-primary:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(102, 126, 234, 0.4);
    background: linear-gradient(135deg, #5a67d8 0%, #6b46c1 100%);
}

.btn-success {
    background: linear-gradient(135deg, #10b981 0%, #059669 100%);
    border: none;
    border-radius: 0.75rem;
    padding: 0.75rem 1.5rem;
    font-weight: 600;
    transition: all 0.3s ease;
    box-shadow: 0 4px 15px rgba(16, 185, 129, 0.3);
}

.btn-success:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(16, 185, 129, 0.4);
    background: linear-gradient(135deg, #059669 0%, #047857 100%);
}

.btn-outline-primary {
    background: transparent;
    border: 2px solid #667eea;
    color: #667eea;
    border-radius: 0.75rem;
    padding: 0.75rem 1.5rem;
    font-weight: 600;
    transition: all 0.3s ease;
}

.btn-outline-primary:hover {
    background: #667eea;
    color: white;
    transform: translateY(-2px);
    box-shadow: 0 4px 15px rgba(102, 126, 234, 0.3);
}

.btn-lg {
    padding: 0.875rem 2rem;
    font-size: 1rem;
}

.form-select {
    width: 100%;
    padding: 0.75rem 1rem;
    border: 2px solid #e5e7eb;
    border-radius: 0.75rem;
    font-size: 0.875rem;
    transition: all 0.3s ease;
    background: white;
}

.form-select:focus {
    outline: none;
    border-color: #667eea;
    box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
}

/* Responsive Design */
@media (max-width: 1024px) {
    .admin-sidebar {
        transform: translateX(-100%);
        transition: transform 0.3s ease;
    }

    .admin-main {
        margin-left: 0;
    }

    .filter-row {
        grid-template-columns: 1fr;
    }

    .analysis-summary {
        grid-template-columns: 1fr;
    }

    .metric-grid {
        grid-template-columns: 1fr;
    }

    .table {
        font-size: 0.875rem;
    }

    .question-preview {
        max-width: 200px;
    }
}

@media (max-width: 768px) {
    .admin-header {
        padding: 1.5rem;
    }

    .admin-header h1 {
        font-size: 1.5rem;
    }

    .header-actions {
        flex-direction: column;
        align-items: stretch;
        gap: 0.5rem;
    }

    .admin-content {
        padding: 1rem;
    }

    .filter-section {
        padding: 1.5rem;
    }

    .summary-card {
        padding: 1rem;
    }

    .stat-icon {
        width: 50px;
        height: 50px;
        font-size: 1.25rem;
    }

    .stat-content h3 {
        font-size: 1.5rem;
    }

    .table-header {
        padding: 1.5rem;
        flex-direction: column;
        align-items: stretch;
        gap: 1rem;
    }

    .loading {
        padding: 3rem 1rem;
    }
}
</style>