// Main JavaScript for public site
document.addEventListener('DOMContentLoaded', function() {
    // Mobile menu toggle
    const mobileMenuButton = document.getElementById('mobileMenuButton');
    const mobileMenu = document.getElementById('mobileMenu');
    const closeMobileMenu = document.getElementById('closeMobileMenu');

    if (mobileMenuButton && mobileMenu) {
        console.log('Mobile menu setup: button found, attaching event listeners');

        // Ensure menu is hidden initially (fallback for non-Tailwind setups)
        if (!mobileMenu.style.transform) {
            mobileMenu.style.transform = 'translateX(-100%)';
        }
        if (!mobileMenu.hasAttribute('aria-hidden')) {
            mobileMenu.setAttribute('aria-hidden', 'true');
        }

        const openMenu = function() {
            mobileMenu.classList.add('show');
            mobileMenu.setAttribute('aria-hidden', 'false');
            mobileMenuButton.setAttribute('aria-expanded', 'true');
        };

        const closeMenu = function() {
            mobileMenu.classList.remove('show');
            mobileMenu.setAttribute('aria-hidden', 'true');
            mobileMenuButton.setAttribute('aria-expanded', 'false');
        };

        const toggleMenu = function(e) {
            e && e.preventDefault();
            e && e.stopPropagation();
            console.log('Toggle menu called');
            const isHidden = !mobileMenu.classList.contains('show');
            console.log('Menu is currently hidden:', isHidden);
            if (isHidden) {
                console.log('Opening menu');
                openMenu();
            } else {
                console.log('Closing menu');
                closeMenu();
            }
            console.log('Menu classes after toggle:', mobileMenu.className);
        };

        // Bind both click and touchstart for better mobile responsiveness
        mobileMenuButton.addEventListener('click', function(e) {
            console.log('Button clicked directly');
            toggleMenu(e);
        }, false);
        mobileMenuButton.addEventListener('touchstart', function(e){ e.preventDefault(); toggleMenu(e); }, {passive:false});

        // Test click handler
        mobileMenuButton.onclick = function() {
            console.log('Direct onclick fired');
        };

        if (closeMobileMenu) {
            closeMobileMenu.addEventListener('click', closeMenu);
            closeMobileMenu.addEventListener('touchstart', function(e){ e.preventDefault(); closeMenu(); }, {passive:false});
        }

        // Close menu when a link or button is clicked
        mobileMenu.querySelectorAll('a, button').forEach(element => {
            element.addEventListener('click', closeMenu);
        });

        // Close menu when clicking outside
        document.addEventListener('click', function(e) {
            if (!mobileMenu.contains(e.target) && !mobileMenuButton.contains(e.target)) {
                closeMenu();
            }
        });

        // Close menu when pressing ESC key
        const handleEscapeKey = function(e) {
            if (e.key === 'Escape' && mobileMenu.classList.contains('show')) {
                closeMenu();
            }
        };
        document.addEventListener('keydown', handleEscapeKey);
    }
    
    // Smooth scrolling for anchor links
    const anchorLinks = document.querySelectorAll('a[href^="#"]');
    
    anchorLinks.forEach(link => {
        link.addEventListener('click', function(e) {
            e.preventDefault();
            
            const targetId = this.getAttribute('href').substring(1);
            const targetElement = document.getElementById(targetId);
            
            if (targetElement) {
                targetElement.scrollIntoView({
                    behavior: 'smooth'
                });
            }
        });
    });
    
    // Header scroll effect
    const header = document.querySelector('.header');
    
    if (header) {
        window.addEventListener('scroll', function() {
            if (window.scrollY > 100) {
                header.style.background = 'rgba(255, 255, 255, 0.98)';
                header.style.boxShadow = '0 4px 6px -1px rgba(0, 0, 0, 0.1)';
            } else {
                header.style.background = 'rgba(255, 255, 255, 0.95)';
                header.style.boxShadow = 'none';
            }
        });
    }
    
    // Animate elements on scroll
    const observerOptions = {
        threshold: 0.1,
        rootMargin: '0px 0px -50px 0px'
    };
    
    const observer = new IntersectionObserver(function(entries) {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                entry.target.style.opacity = '1';
                entry.target.style.transform = 'translateY(0)';
            }
        });
    }, observerOptions);
    
    // Observe post cards and category cards
    const animatedElements = document.querySelectorAll('.post-card, .category-card');
    
    animatedElements.forEach(el => {
        el.style.opacity = '0';
        el.style.transform = 'translateY(20px)';
        el.style.transition = 'opacity 0.6s ease, transform 0.6s ease';
        observer.observe(el);
    });
});
