<?php
require_once '../../config/database.php';
require_once '../../config/auth.php';
require_once '../../config/rbac.php';
require_once '../../includes/functions.php';
require_once '../../includes/middleware.php';

// Check if user is instructor or admin
anyRoleMiddleware(['instructor', 'admin']);

$user = getUserContext();
$db = getDB();
$userId = $_SESSION['user_id'];

// Get exam ID from URL
$examId = $_GET['id'] ?? null;
if (!$examId) {
    header('Location: index.php');
    exit;
}

// Get exam details
$stmt = $db->prepare("
    SELECT e.*, c.title as course_title
    FROM exams e
    LEFT JOIN courses c ON e.course_id = c.id
    WHERE e.id = ? AND e.instructor_id = ?
");
$stmt->execute([$examId, $userId]);
$exam = $stmt->fetch();

if (!$exam) {
    $_SESSION['error_message'] = 'Exam not found or access denied';
    header('Location: index.php');
    exit;
}

// Get exam attempts with student info
$stmt = $db->prepare("
    SELECT ea.*, u.username, u.email, up.first_name, up.last_name,
           TIMESTAMPDIFF(MINUTE, ea.started_at, ea.completed_at) as time_taken
    FROM exam_attempts ea
    JOIN users u ON ea.student_id = u.id
    LEFT JOIN user_profiles up ON u.id = up.user_id
    WHERE ea.exam_id = ?
    ORDER BY ea.completed_at DESC, ea.started_at DESC
");
$stmt->execute([$examId]);
$attempts = $stmt->fetchAll();

// Calculate statistics
$totalAttempts = count($attempts);
$completedAttempts = count(array_filter($attempts, fn($a) => $a['status'] === 'completed'));
$passedAttempts = count(array_filter($attempts, fn($a) => $a['is_passed'] == 1));
$averageScore = $completedAttempts > 0 ? array_sum(array_column(array_filter($attempts, fn($a) => $a['status'] === 'completed'), 'percentage')) / $completedAttempts : 0;

$page_title = 'Exam Results - ' . htmlspecialchars($exam['title']);
include '../../includes/header.php';
?>

<div class="admin-container">
    <div class="admin-sidebar">
        <div class="sidebar-header">
            <h2><i class="fas fa-graduation-cap me-2"></i>Instructor Panel</h2>
            <p>Instructor Panel</p>
        </div>

        <nav class="sidebar-nav">
            <ul>
                <li><a href="../dashboard.php"><i class="fas fa-tachometer-alt me-2"></i>Dashboard</a></li>
                <li><a href="../courses/"><i class="fas fa-book me-2"></i>My Courses</a></li>
                <li><a href="../courses/new.php"><i class="fas fa-plus me-2"></i>Create Course</a></li>
                <li><a href="../questions/index.php"><i class="fas fa-question-circle me-2"></i>Question Bank</a></li>
                <li><a href="index.php" class="active"><i class="fas fa-file-alt me-2"></i>Exams</a></li>
                <li><a href="assignments.php"><i class="fas fa-user-plus me-2"></i>Exam Assignments</a></li>
                <li><a href="../resources.php"><i class="fas fa-folder me-2"></i>Resources</a></li>
                <li><a href="../students.php"><i class="fas fa-users me-2"></i>Students</a></li>
                <li><a href="../assignments.php"><i class="fas fa-tasks me-2"></i>Assignments</a></li>
                <li><a href="/lms/instructor/grading/index.php"><i class="fas fa-chart-line me-2"></i>Grades</a></li>
                <li><a href="../../logout.php"><i class="fas fa-sign-out-alt me-2"></i>Logout</a></li>
            </ul>
        </nav>
    </div>

    <div class="admin-main">
        <div class="admin-header">
            <div class="d-flex justify-content-between align-items-center">
                <div>
                    <h1><i class="fas fa-chart-bar text-success me-3"></i>Exam Results</h1>
                    <p class="text-muted mb-0"><?php echo htmlspecialchars($exam['title']); ?> - Student Performance</p>
                </div>
                <div class="header-actions">
                    <a href="edit.php?id=<?php echo $exam['id']; ?>" class="btn btn-warning btn-lg">
                        <i class="fas fa-edit me-2"></i>Edit Exam
                    </a>
                    <a href="preview.php?id=<?php echo $exam['id']; ?>" class="btn btn-info btn-lg">
                        <i class="fas fa-eye me-2"></i>Preview Exam
                    </a>
                    <a href="index.php" class="btn btn-outline-secondary btn-lg">
                        <i class="fas fa-arrow-left me-2"></i>Back to Exams
                    </a>
                </div>
            </div>
        </div>

        <div class="admin-content">
            <!-- Exam Overview -->
            <div class="exam-overview">
                <div class="overview-card">
                    <h3><?php echo htmlspecialchars($exam['title']); ?></h3>
                    <div class="exam-meta">
                        <div class="meta-item">
                            <i class="fas fa-users"></i>
                            <span><?php echo $totalAttempts; ?> Total Attempts</span>
                        </div>
                        <div class="meta-item">
                            <i class="fas fa-check-circle"></i>
                            <span><?php echo $completedAttempts; ?> Completed</span>
                        </div>
                        <div class="meta-item">
                            <i class="fas fa-trophy"></i>
                            <span><?php echo $passedAttempts; ?> Passed (<?php echo $completedAttempts > 0 ? round(($passedAttempts / $completedAttempts) * 100, 1) : 0; ?>%)</span>
                        </div>
                        <div class="meta-item">
                            <i class="fas fa-chart-line"></i>
                            <span><?php echo round($averageScore, 1); ?>% Average Score</span>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Results Table -->
            <div class="results-container">
                <div class="results-header">
                    <h3><i class="fas fa-list me-2"></i>Student Attempts</h3>
                </div>

                <?php if (empty($attempts)): ?>
                    <div class="empty-state">
                        <div class="empty-state-icon">
                            <i class="fas fa-chart-bar"></i>
                        </div>
                        <h3>No Attempts Yet</h3>
                        <p>Students haven't taken this exam yet. Results will appear here once they do.</p>
                    </div>
                <?php else: ?>
                    <div class="results-table-container">
                        <table class="results-table">
                            <thead>
                                <tr>
                                    <th>Student</th>
                                    <th>Attempt</th>
                                    <th>Status</th>
                                    <th>Score</th>
                                    <th>Percentage</th>
                                    <th>Result</th>
                                    <th>Time Taken</th>
                                    <th>Completed</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($attempts as $attempt): ?>
                                    <tr>
                                        <td>
                                            <div class="student-info">
                                                <div class="student-name">
                                                    <?php echo htmlspecialchars($attempt['first_name'] . ' ' . $attempt['last_name']); ?>
                                                </div>
                                                <div class="student-email">
                                                    <?php echo htmlspecialchars($attempt['email']); ?>
                                                </div>
                                            </div>
                                        </td>
                                        <td><?php echo $attempt['attempt_number']; ?></td>
                                        <td>
                                            <span class="status-badge <?php
                                                echo $attempt['status'] === 'completed' ? 'completed' :
                                                     ($attempt['status'] === 'in_progress' ? 'in-progress' : 'abandoned');
                                            ?>">
                                                <?php echo ucfirst(str_replace('_', ' ', $attempt['status'])); ?>
                                            </span>
                                        </td>
                                        <td>
                                            <?php if ($attempt['status'] === 'completed'): ?>
                                                <?php echo $attempt['score']; ?>/<?php echo $attempt['max_score']; ?>
                                            <?php else: ?>
                                                <span class="text-muted">-</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php if ($attempt['status'] === 'completed'): ?>
                                                <span class="percentage <?php echo $attempt['percentage'] >= $exam['passing_score'] ? 'pass' : 'fail'; ?>">
                                                    <?php echo round($attempt['percentage'], 1); ?>%
                                                </span>
                                            <?php else: ?>
                                                <span class="text-muted">-</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php if ($attempt['status'] === 'completed'): ?>
                                                <span class="result-badge <?php echo $attempt['is_passed'] ? 'passed' : 'failed'; ?>">
                                                    <?php echo $attempt['is_passed'] ? 'PASSED' : 'FAILED'; ?>
                                                </span>
                                            <?php else: ?>
                                                <span class="text-muted">-</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php if ($attempt['status'] === 'completed' && $attempt['time_taken']): ?>
                                                <?php echo $attempt['time_taken']; ?> min
                                            <?php else: ?>
                                                <span class="text-muted">-</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php if ($attempt['completed_at']): ?>
                                                <?php echo date('M j, Y g:i A', strtotime($attempt['completed_at'])); ?>
                                            <?php else: ?>
                                                <span class="text-muted">-</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php if ($attempt['status'] === 'completed'): ?>
                                                <button class="btn btn-sm btn-outline-primary" onclick="viewAttemptDetails(<?php echo $attempt['id']; ?>)">
                                                    <i class="fas fa-eye"></i> View
                                                </button>
                                            <?php else: ?>
                                                <span class="text-muted">-</span>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<!-- Attempt Details Modal -->
<div id="attemptModal" class="modal-overlay" style="display: none;">
    <div class="modal-content">
        <div class="modal-header">
            <h3 id="modalTitle">Attempt Details</h3>
            <button type="button" class="modal-close" onclick="closeModal()">
                <i class="fas fa-times"></i>
            </button>
        </div>
        <div class="modal-body" id="modalBody">
            <!-- Attempt details will be loaded here -->
        </div>
    </div>
</div>

<script>
function viewAttemptDetails(attemptId) {
    // This would typically load attempt details via AJAX
    // For now, we'll show a placeholder
    document.getElementById('modalTitle').textContent = 'Attempt Details';
    document.getElementById('modalBody').innerHTML = `
        <div class="text-center p-4">
            <i class="fas fa-info-circle text-4xl text-blue-500 mb-4"></i>
            <p>Detailed attempt view would be implemented here.</p>
            <p>Attempt ID: ${attemptId}</p>
        </div>
    `;
    document.getElementById('attemptModal').style.display = 'flex';
}

function closeModal() {
    document.getElementById('attemptModal').style.display = 'none';
}

// Close modal when clicking outside
document.getElementById('attemptModal').addEventListener('click', function(e) {
    if (e.target === this) {
        closeModal();
    }
});
</script>

<style>
/* Custom styles for Exam Results page */
.admin-container {
    display: flex;
    min-height: 100vh;
    background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
}

.admin-sidebar {
    width: 280px;
    background: linear-gradient(135deg, #1e293b 0%, #334155 100%);
    color: white;
    position: fixed;
    height: 100vh;
    overflow-y: auto;
    box-shadow: 4px 0 15px rgba(0, 0, 0, 0.1);
    z-index: 1000;
}

.sidebar-header {
    padding: 2rem 1.5rem;
    border-bottom: 1px solid #374151;
    text-align: center;
}

.sidebar-header h2 {
    font-size: 1.5rem;
    font-weight: 800;
    margin-bottom: 0.5rem;
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 0.5rem;
}

.sidebar-nav ul {
    list-style: none;
    padding: 0;
    margin: 0;
}

.sidebar-nav li a {
    display: flex;
    align-items: center;
    gap: 0.75rem;
    padding: 1rem 1.5rem;
    color: #cbd5e1;
    text-decoration: none;
    transition: all 0.3s ease;
    font-weight: 500;
    border-left: 3px solid transparent;
}

.sidebar-nav li a:hover,
.sidebar-nav li a.active {
    background: linear-gradient(135deg, #374151 0%, #4b5563 100%);
    color: white;
    border-left-color: #3b82f6;
    transform: translateX(5px);
}

.admin-main {
    margin-left: 280px;
    flex: 1;
    min-height: 100vh;
}

.admin-header {
    background: white;
    padding: 2rem;
    border-bottom: 1px solid #e5e7eb;
    box-shadow: 0 2px 10px rgba(0, 0, 0, 0.05);
}

.admin-header h1 {
    font-size: 2rem;
    font-weight: 800;
    color: #1f2937;
    margin-bottom: 0.5rem;
    display: flex;
    align-items: center;
    gap: 0.75rem;
}

.header-actions {
    display: flex;
    gap: 1rem;
    align-items: center;
}

.admin-content {
    padding: 2rem;
}

.exam-overview {
    margin-bottom: 2rem;
}

.overview-card {
    background: white;
    border-radius: 1rem;
    padding: 2rem;
    box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
    border: 1px solid rgba(255, 255, 255, 0.8);
}

.overview-card h3 {
    font-size: 1.5rem;
    font-weight: 700;
    color: #1f2937;
    margin-bottom: 1.5rem;
}

.exam-meta {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 1rem;
}

.meta-item {
    display: flex;
    align-items: center;
    gap: 0.75rem;
    padding: 1rem;
    background: #f8fafc;
    border-radius: 0.5rem;
    border: 1px solid #e5e7eb;
}

.meta-item i {
    color: #3b82f6;
    font-size: 1.1rem;
}

.meta-item span {
    font-weight: 600;
    color: #374151;
}

.results-container {
    background: white;
    border-radius: 1rem;
    box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
    border: 1px solid rgba(255, 255, 255, 0.8);
    overflow: hidden;
}

.results-header {
    padding: 2rem;
    background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
    border-bottom: 1px solid #e5e7eb;
}

.results-header h3 {
    font-size: 1.5rem;
    font-weight: 700;
    color: #1f2937;
    margin: 0;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.results-table-container {
    overflow-x: auto;
}

.results-table {
    width: 100%;
    border-collapse: collapse;
}

.results-table th,
.results-table td {
    padding: 1rem;
    text-align: left;
    border-bottom: 1px solid #e5e7eb;
}

.results-table th {
    background: #f8fafc;
    font-weight: 600;
    color: #374151;
    font-size: 0.875rem;
    text-transform: uppercase;
    letter-spacing: 0.05em;
}

.results-table tbody tr:hover {
    background: #f8fafc;
}

.student-info {
    display: flex;
    flex-direction: column;
}

.student-name {
    font-weight: 600;
    color: #1f2937;
}

.student-email {
    font-size: 0.875rem;
    color: #6b7280;
}

.status-badge {
    display: inline-flex;
    align-items: center;
    padding: 0.25rem 0.75rem;
    border-radius: 1rem;
    font-size: 0.75rem;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.05em;
}

.status-badge.completed {
    background: linear-gradient(135deg, #10b981 0%, #059669 100%);
    color: white;
}

.status-badge.in-progress {
    background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%);
    color: white;
}

.status-badge.abandoned {
    background: linear-gradient(135deg, #6b7280 0%, #4b5563 100%);
    color: white;
}

.percentage.pass {
    color: #059669;
    font-weight: 600;
}

.percentage.fail {
    color: #dc2626;
    font-weight: 600;
}

.result-badge {
    display: inline-flex;
    align-items: center;
    padding: 0.25rem 0.75rem;
    border-radius: 1rem;
    font-size: 0.75rem;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.05em;
}

.result-badge.passed {
    background: linear-gradient(135deg, #10b981 0%, #059669 100%);
    color: white;
}

.result-badge.failed {
    background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%);
    color: white;
}

.btn {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.5rem 1rem;
    font-weight: 600;
    text-decoration: none;
    border-radius: 0.5rem;
    border: none;
    cursor: pointer;
    transition: all 0.3s ease;
    font-size: 0.875rem;
}

.btn-sm {
    padding: 0.375rem 0.75rem;
    font-size: 0.75rem;
}

.btn-outline-primary {
    background: transparent;
    border: 2px solid #3b82f6;
    color: #3b82f6;
}

.btn-outline-primary:hover {
    background: #3b82f6;
    color: white;
    transform: translateY(-1px);
    box-shadow: 0 4px 15px rgba(59, 130, 246, 0.3);
}

.btn-warning {
    background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%);
    color: white;
    box-shadow: 0 4px 15px rgba(245, 158, 11, 0.3);
}

.btn-warning:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(245, 158, 11, 0.4);
    background: linear-gradient(135deg, #d97706 0%, #b45309 100%);
}

.btn-info {
    background: linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%);
    color: white;
    box-shadow: 0 4px 15px rgba(59, 130, 246, 0.3);
}

.btn-info:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(59, 130, 246, 0.4);
    background: linear-gradient(135deg, #2563eb 0%, #1e40af 100%);
}

.btn-outline-secondary {
    background: transparent;
    border: 2px solid #6b7280;
    color: #6b7280;
}

.btn-outline-secondary:hover {
    background: #6b7280;
    color: white;
    transform: translateY(-2px);
    box-shadow: 0 4px 15px rgba(107, 114, 128, 0.3);
}

.btn-lg {
    padding: 0.875rem 2rem;
    font-size: 1rem;
}

.text-muted {
    color: #6b7280;
}

.empty-state {
    text-align: center;
    padding: 4rem 2rem;
    background: white;
    border-radius: 1rem;
    box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
}

.empty-state-icon {
    width: 80px;
    height: 80px;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    margin: 0 auto 2rem;
    color: white;
    font-size: 2rem;
    box-shadow: 0 8px 32px rgba(102, 126, 234, 0.3);
}

.empty-state h3 {
    font-size: 1.5rem;
    font-weight: 700;
    color: #1f2937;
    margin-bottom: 1rem;
}

.empty-state p {
    color: #64748b;
    font-size: 1.1rem;
    margin-bottom: 2rem;
}

/* Modal Styles */
.modal-overlay {
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: rgba(0, 0, 0, 0.5);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 1000;
}

.modal-content {
    background: white;
    border-radius: 1rem;
    box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
    max-width: 90vw;
    max-height: 90vh;
    width: 600px;
    overflow: hidden;
    display: flex;
    flex-direction: column;
}

.modal-header {
    padding: 1.5rem 2rem;
    background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
    border-bottom: 1px solid #e5e7eb;
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.modal-header h3 {
    font-size: 1.25rem;
    font-weight: 700;
    color: #1f2937;
    margin: 0;
}

.modal-close {
    background: none;
    border: none;
    color: #6b7280;
    font-size: 1.25rem;
    cursor: pointer;
    padding: 0.5rem;
    border-radius: 0.5rem;
    transition: all 0.3s ease;
}

.modal-close:hover {
    background: #f3f4f6;
    color: #374151;
}

.modal-body {
    padding: 2rem;
    overflow-y: auto;
    flex: 1;
}

/* Responsive Design */
@media (max-width: 1024px) {
    .admin-sidebar {
        transform: translateX(-100%);
        transition: transform 0.3s ease;
    }

    .admin-main {
        margin-left: 0;
    }

    .exam-meta {
        grid-template-columns: 1fr;
    }

    .results-table {
        font-size: 0.875rem;
    }

    .results-table th,
    .results-table td {
        padding: 0.75rem 0.5rem;
    }
}

@media (max-width: 768px) {
    .admin-header {
        padding: 1.5rem;
    }

    .admin-header h1 {
        font-size: 1.5rem;
    }

    .header-actions {
        flex-direction: column;
        align-items: stretch;
        gap: 0.5rem;
    }

    .admin-content {
        padding: 1rem;
    }

    .overview-card {
        padding: 1.5rem;
    }

    .results-header {
        padding: 1.5rem;
    }

    .results-table-container {
        overflow-x: scroll;
    }

    .empty-state {
        padding: 3rem 1rem;
    }

    .empty-state-icon {
        width: 60px;
        height: 60px;
        font-size: 1.5rem;
    }

    .modal-content {
        width: 95vw;
        margin: 1rem;
    }
}
</style>

<?php include '../../includes/footer.php'; ?>