<?php
require_once '../../config/database.php';
require_once '../../config/auth.php';
require_once '../../config/rbac.php';
require_once '../../includes/functions.php';
require_once '../../includes/middleware.php';

// Check if user is instructor or admin
anyRoleMiddleware(['instructor', 'admin']);

$user = getUserContext();
$db = getDB();
$userId = $_SESSION['user_id'];

// Get question ID from URL
$questionId = $_GET['id'] ?? null;
if (!$questionId) {
    header('Location: index.php');
    exit;
}

// Get question details
$stmt = $db->prepare("
    SELECT q.*, qc.name as category_name, c.title as course_title
    FROM questions q
    LEFT JOIN question_categories qc ON q.category_id = qc.id
    LEFT JOIN courses c ON q.course_id = c.id
    WHERE q.id = ? AND (q.instructor_id = ? OR q.is_shared = TRUE)
");
$stmt->execute([$questionId, $userId]);
$question = $stmt->fetch();

if (!$question) {
    $_SESSION['error_message'] = 'Question not found or access denied';
    header('Location: index.php');
    exit;
}

// Get question options
$stmt = $db->prepare("
    SELECT * FROM question_options
    WHERE question_id = ?
    ORDER BY sort_order
");
$stmt->execute([$questionId]);
$options = $stmt->fetchAll();

$page_title = 'Preview Question - ' . htmlspecialchars(substr($question['title'], 0, 50));
include '../../includes/header.php';
?>

<div class="admin-container">
    <div class="admin-sidebar">
        <div class="sidebar-header">
            <h2><i class="fas fa-graduation-cap me-2"></i>Instructor Panel</h2>
            <p>Instructor Panel</p>
        </div>

        <nav class="sidebar-nav">
            <ul>
                <li><a href="../dashboard.php"><i class="fas fa-tachometer-alt me-2"></i>Dashboard</a></li>
                <li><a href="../courses/"><i class="fas fa-book me-2"></i>My Courses</a></li>
                <li><a href="../courses/new.php"><i class="fas fa-plus me-2"></i>Create Course</a></li>
                <li><a href="index.php" class="active"><i class="fas fa-question-circle me-2"></i>Question Bank</a></li>
                <li><a href="../exams/"><i class="fas fa-file-alt me-2"></i>Exams</a></li>
                <li><a href="../resources.php"><i class="fas fa-folder me-2"></i>Resources</a></li>
                <li><a href="../students.php"><i class="fas fa-users me-2"></i>Students</a></li>
                <li><a href="../assignments.php"><i class="fas fa-tasks me-2"></i>Assignments</a></li>
                <li><a href="/lms/instructor/grading/index.php"><i class="fas fa-chart-line me-2"></i>Grades</a></li>
                <li><a href="../../logout.php"><i class="fas fa-sign-out-alt me-2"></i>Logout</a></li>
            </ul>
        </nav>
    </div>

    <div class="admin-main">
        <div class="admin-header">
            <div class="d-flex justify-content-between align-items-center">
                <div>
                    <h1><i class="fas fa-eye text-info me-3"></i>Preview Question</h1>
                    <p class="text-muted mb-0"><?php echo htmlspecialchars($question['title']); ?> - Read-only preview</p>
                </div>
                <div class="header-actions">
                    <a href="edit.php?id=<?php echo $question['id']; ?>" class="btn btn-warning btn-lg">
                        <i class="fas fa-edit me-2"></i>Edit Question
                    </a>
                    <a href="index.php" class="btn btn-outline-secondary btn-lg">
                        <i class="fas fa-arrow-left me-2"></i>Back to Questions
                    </a>
                </div>
            </div>
        </div>

        <div class="admin-content">
            <!-- Question Details -->
            <div class="question-preview-header">
                <div class="question-info-card">
                    <h3><?php echo htmlspecialchars($question['title']); ?></h3>
                    <?php if ($question['content']): ?>
                        <div class="question-content-preview">
                            <?php echo nl2br(htmlspecialchars($question['content'])); ?>
                        </div>
                    <?php endif; ?>

                    <div class="question-meta">
                        <div class="meta-item">
                            <i class="fas fa-cogs"></i>
                            <span><?php echo ucfirst(str_replace('_', ' ', $question['question_type'])); ?></span>
                        </div>
                        <div class="meta-item">
                            <i class="fas fa-chart-line"></i>
                            <span><?php echo ucfirst($question['difficulty']); ?> difficulty</span>
                        </div>
                        <div class="meta-item">
                            <i class="fas fa-tag"></i>
                            <span><?php echo htmlspecialchars($question['category_name'] ?? 'Uncategorized'); ?></span>
                        </div>
                        <div class="meta-item">
                            <i class="fas fa-info-circle"></i>
                            <span><?php echo ucfirst(str_replace('_', ' ', $question['status'])); ?></span>
                        </div>
                        <?php if ($question['course_title']): ?>
                        <div class="meta-item">
                            <i class="fas fa-book"></i>
                            <span><?php echo htmlspecialchars($question['course_title']); ?></span>
                        </div>
                        <?php endif; ?>
                    </div>

                    <?php if ($question['explanation']): ?>
                        <div class="question-explanation">
                            <h4>Explanation:</h4>
                            <p><?php echo nl2br(htmlspecialchars($question['explanation'])); ?></p>
                        </div>
                    <?php endif; ?>
                </div>
            </div>

            <!-- Question Options -->
            <div class="question-options-preview">
                <div class="question-preview-card">
                    <div class="question-header">
                        <h4>Question Preview</h4>
                        <div class="question-meta">
                            <span class="question-type"><?php echo ucfirst(str_replace('_', ' ', $question['question_type'])); ?></span>
                        </div>
                    </div>

                    <div class="question-content">
                        <div class="question-text">
                            <?php echo nl2br(htmlspecialchars($question['content'])); ?>
                        </div>

                        <div class="question-options">
                            <?php if (in_array($question['question_type'], ['mcq_single', 'true_false'])): ?>
                                <?php foreach ($options as $option): ?>
                                    <div class="option-item <?php echo $option['is_correct'] ? 'correct' : ''; ?>">
                                        <div class="option-radio">
                                            <i class="far fa-<?php echo $option['is_correct'] ? 'dot-circle' : 'circle'; ?>"></i>
                                        </div>
                                        <div class="option-text">
                                            <?php echo htmlspecialchars($option['option_text']); ?>
                                            <?php if ($option['is_correct']): ?>
                                                <span class="correct-indicator">(Correct Answer)</span>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                <?php endforeach; ?>

                            <?php elseif ($question['question_type'] === 'mcq_multiple'): ?>
                                <?php foreach ($options as $option): ?>
                                    <div class="option-item <?php echo $option['is_correct'] ? 'correct' : ''; ?>">
                                        <div class="option-checkbox">
                                            <i class="far fa-<?php echo $option['is_correct'] ? 'check-square' : 'square'; ?>"></i>
                                        </div>
                                        <div class="option-text">
                                            <?php echo htmlspecialchars($option['option_text']); ?>
                                            <?php if ($option['is_correct']): ?>
                                                <span class="correct-indicator">(Correct Answer)</span>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                <?php endforeach; ?>

                            <?php elseif (in_array($question['question_type'], ['essay', 'short_answer'])): ?>
                                <div class="text-answer-placeholder">
                                    <textarea readonly rows="<?php echo $question['question_type'] === 'essay' ? 6 : 3; ?>"
                                              placeholder="Student answer area"></textarea>
                                </div>

                            <?php elseif ($question['question_type'] === 'code'): ?>
                                <div class="code-answer-placeholder">
                                    <textarea readonly rows="8" placeholder="Code answer area"></textarea>
                                </div>

                            <?php elseif ($question['question_type'] === 'matching'): ?>
                                <div class="matching-question">
                                    <div class="matching-container">
                                        <div class="matching-columns">
                                            <div class="matching-column">
                                                <h5 class="matching-column-title">Terms</h5>
                                                <div class="matching-items">
                                                    <?php foreach ($options as $index => $option): ?>
                                                        <div class="matching-item" data-index="<?php echo $index; ?>">
                                                            <div class="matching-item-content">
                                                                <?php echo htmlspecialchars($option['option_text']); ?>
                                                            </div>
                                                            <div class="matching-connector"></div>
                                                        </div>
                                                    <?php endforeach; ?>
                                                </div>
                                            </div>
                                            <div class="matching-column">
                                                <h5 class="matching-column-title">Definitions</h5>
                                                <div class="matching-items">
                                                    <?php foreach ($options as $index => $option): ?>
                                                        <div class="matching-item" data-index="<?php echo $index; ?>">
                                                            <div class="matching-connector"></div>
                                                            <div class="matching-item-content">
                                                                <?php echo htmlspecialchars($option['option_value']); ?>
                                                            </div>
                                                        </div>
                                                    <?php endforeach; ?>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>

                            <?php elseif ($question['question_type'] === 'fill_blanks'): ?>
                                <div class="fill-blanks-placeholder">
                                    <p>Fill in the blanks question would display here.</p>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<style>
/* Custom styles for Question Preview page */
.admin-container {
    display: flex;
    min-height: 100vh;
    background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
}

.admin-sidebar {
    width: 280px;
    background: linear-gradient(135deg, #1e293b 0%, #334155 100%);
    color: white;
    position: fixed;
    height: 100vh;
    overflow-y: auto;
    box-shadow: 4px 0 15px rgba(0, 0, 0, 0.1);
    z-index: 1000;
}

.sidebar-header {
    padding: 2rem 1.5rem;
    border-bottom: 1px solid #374151;
    text-align: center;
}

.sidebar-header h2 {
    font-size: 1.5rem;
    font-weight: 800;
    margin-bottom: 0.5rem;
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 0.5rem;
}

.sidebar-nav ul {
    list-style: none;
    padding: 0;
    margin: 0;
}

.sidebar-nav li a {
    display: flex;
    align-items: center;
    gap: 0.75rem;
    padding: 1rem 1.5rem;
    color: #cbd5e1;
    text-decoration: none;
    transition: all 0.3s ease;
    font-weight: 500;
    border-left: 3px solid transparent;
}

.sidebar-nav li a:hover,
.sidebar-nav li a.active {
    background: linear-gradient(135deg, #374151 0%, #4b5563 100%);
    color: white;
    border-left-color: #3b82f6;
    transform: translateX(5px);
}

.admin-main {
    margin-left: 280px;
    flex: 1;
    min-height: 100vh;
}

.admin-header {
    background: white;
    padding: 2rem;
    border-bottom: 1px solid #e5e7eb;
    box-shadow: 0 2px 10px rgba(0, 0, 0, 0.05);
}

.admin-header h1 {
    font-size: 2rem;
    font-weight: 800;
    color: #1f2937;
    margin-bottom: 0.5rem;
    display: flex;
    align-items: center;
    gap: 0.75rem;
}

.header-actions {
    display: flex;
    gap: 1rem;
    align-items: center;
}

.admin-content {
    padding: 2rem;
}

.question-preview-header {
    margin-bottom: 2rem;
}

.question-info-card {
    background: white;
    border-radius: 1rem;
    padding: 2rem;
    box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
    border: 1px solid rgba(255, 255, 255, 0.8);
}

.question-info-card h3 {
    font-size: 1.5rem;
    font-weight: 700;
    color: #1f2937;
    margin-bottom: 1rem;
}

.question-content-preview {
    color: #64748b;
    font-size: 1rem;
    line-height: 1.6;
    margin-bottom: 1.5rem;
    padding: 1rem;
    background: #f8fafc;
    border-radius: 0.5rem;
    border: 1px solid #e5e7eb;
}

.question-meta {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 1rem;
    margin-bottom: 1.5rem;
}

.meta-item {
    display: flex;
    align-items: center;
    gap: 0.75rem;
    padding: 0.75rem;
    background: #f8fafc;
    border-radius: 0.5rem;
    border: 1px solid #e5e7eb;
}

.meta-item i {
    color: #3b82f6;
    font-size: 1.1rem;
}

.meta-item span {
    font-weight: 600;
    color: #374151;
}

.question-explanation {
    background: #fef3c7;
    border: 1px solid #f59e0b;
    border-radius: 0.5rem;
    padding: 1rem;
}

.question-explanation h4 {
    font-weight: 600;
    color: #92400e;
    margin-bottom: 0.5rem;
}

.question-explanation p {
    color: #78350f;
    margin: 0;
}

.question-options-preview {
    display: flex;
    flex-direction: column;
    gap: 1.5rem;
}

.question-preview-card {
    background: white;
    border-radius: 1rem;
    box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
    border: 1px solid rgba(255, 255, 255, 0.8);
    overflow: hidden;
}

.question-header {
    padding: 1.5rem 2rem;
    background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
    border-bottom: 1px solid #e5e7eb;
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.question-header h4 {
    font-size: 1.25rem;
    font-weight: 700;
    color: #1f2937;
    margin: 0;
}

.question-type {
    background: linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%);
    color: white;
    padding: 0.25rem 0.75rem;
    border-radius: 1rem;
    font-size: 0.75rem;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.05em;
}

.question-content {
    padding: 2rem;
}

.question-text {
    margin-bottom: 1.5rem;
    font-size: 1rem;
    line-height: 1.6;
    color: #374151;
}

.question-options {
    display: flex;
    flex-direction: column;
    gap: 1rem;
}

.option-item {
    display: flex;
    align-items: flex-start;
    gap: 1rem;
    padding: 1rem;
    background: #f8fafc;
    border-radius: 0.5rem;
    border: 1px solid #e5e7eb;
    transition: all 0.3s ease;
}

.option-item.correct {
    background: #f0fdf4;
    border-color: #10b981;
}

.option-radio,
.option-checkbox {
    color: #6b7280;
    font-size: 1.1rem;
    margin-top: 0.125rem;
}

.option-item.correct .option-radio,
.option-item.correct .option-checkbox {
    color: #10b981;
}

.option-text {
    flex: 1;
    color: #374151;
}

.correct-indicator {
    color: #10b981;
    font-weight: 600;
    font-size: 0.875rem;
    margin-left: 0.5rem;
}

.text-answer-placeholder textarea,
.code-answer-placeholder textarea {
    width: 100%;
    padding: 1rem;
    border: 2px solid #e5e7eb;
    border-radius: 0.5rem;
    background: #f9fafb;
    color: #6b7280;
    font-family: inherit;
    resize: vertical;
}

.code-answer-placeholder textarea {
    font-family: 'Courier New', monospace;
}

.matching-question {
    padding: 1.5rem;
    background: #f8fafc;
    border-radius: 0.5rem;
    border: 1px solid #e5e7eb;
}

.matching-container {
    max-width: 100%;
}

.matching-columns {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 2rem;
    align-items: start;
}

.matching-column {
    background: white;
    border-radius: 0.5rem;
    padding: 1.5rem;
    box-shadow: 0 1px 3px rgba(0, 0, 0, 0.1);
}

.matching-column-title {
    font-size: 1rem;
    font-weight: 600;
    color: #374151;
    margin-bottom: 1rem;
    text-align: center;
    border-bottom: 2px solid #e5e7eb;
    padding-bottom: 0.5rem;
}

.matching-items {
    display: flex;
    flex-direction: column;
    gap: 1rem;
}

.matching-item {
    display: flex;
    align-items: center;
    gap: 0.75rem;
    padding: 0.75rem;
    background: #f9fafb;
    border-radius: 0.375rem;
    border: 1px solid #e5e7eb;
    position: relative;
}

.matching-item-content {
    flex: 1;
    font-size: 0.875rem;
    color: #374151;
    line-height: 1.4;
}

.matching-connector {
    width: 12px;
    height: 2px;
    background: #10b981;
    border-radius: 1px;
    position: relative;
}

.matching-connector::before {
    content: '';
    position: absolute;
    left: 50%;
    top: 50%;
    transform: translate(-50%, -50%);
    width: 6px;
    height: 6px;
    background: #10b981;
    border-radius: 50%;
}

.matching-placeholder,
.fill-blanks-placeholder {
    padding: 2rem;
    text-align: center;
    color: #6b7280;
    font-style: italic;
}

.btn {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.75rem 1.5rem;
    font-weight: 600;
    text-decoration: none;
    border-radius: 0.75rem;
    border: none;
    cursor: pointer;
    transition: all 0.3s ease;
    font-size: 0.875rem;
}

.btn-warning {
    background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%);
    color: white;
    box-shadow: 0 4px 15px rgba(245, 158, 11, 0.3);
}

.btn-warning:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(245, 158, 11, 0.4);
    background: linear-gradient(135deg, #d97706 0%, #b45309 100%);
}

.btn-outline-secondary {
    background: transparent;
    border: 2px solid #6b7280;
    color: #6b7280;
}

.btn-outline-secondary:hover {
    background: #6b7280;
    color: white;
    transform: translateY(-2px);
    box-shadow: 0 4px 15px rgba(107, 114, 128, 0.3);
}

.btn-lg {
    padding: 0.875rem 2rem;
    font-size: 1rem;
}

/* Responsive Design */
@media (max-width: 1024px) {
    .admin-sidebar {
        transform: translateX(-100%);
        transition: transform 0.3s ease;
    }

    .admin-main {
        margin-left: 0;
    }

    .question-meta {
        grid-template-columns: 1fr;
    }

    .question-header {
        flex-direction: column;
        align-items: stretch;
        gap: 1rem;
    }
}

@media (max-width: 768px) {
    .admin-header {
        padding: 1.5rem;
    }

    .admin-header h1 {
        font-size: 1.5rem;
    }

    .header-actions {
        flex-direction: column;
        align-items: stretch;
        gap: 0.5rem;
    }

    .admin-content {
        padding: 1rem;
    }

    .question-info-card {
        padding: 1.5rem;
    }

    .question-content {
        padding: 1.5rem;
    }
}
</style>

<?php include '../../includes/footer.php'; ?>