<?php
require_once '../includes/middleware.php';
require_once '../config/database.php';
require_once '../config/currency.php';
require_once '../includes/functions.php';

// Cart Manager Class (copied from api/cart.php)
class CartManager {
    private $db;
    private $userId;

    public function __construct($db, $userId = null) {
        $this->db = $db;
        $this->userId = $userId;
    }

    public function getCart() {
        if (!$this->userId) {
            return ['items' => [], 'total' => 0.00, 'currency' => 'USD'];
        }

        // Get cart items from database (assuming cart_items table exists)
        $stmt = $this->db->prepare("
            SELECT ci.*, c.title, c.price, c.currency, c.is_free, c.thumbnail, c.max_students
            FROM cart_items ci
            JOIN courses c ON ci.item_id = c.id
            WHERE ci.user_id = ? AND ci.item_type = 'course'
            ORDER BY ci.created_at ASC
        ");
        $stmt->execute([$this->userId]);
        $items = $stmt->fetchAll(PDO::FETCH_ASSOC);

        $total = 0.00;
        $currency = 'USD';

        foreach ($items as &$item) {
            if (!$item['is_free']) {
                $total += floatval($item['price']);
                $currency = $item['currency'];
            }
        }

        return [
            'items' => $items,
            'total' => $total,
            'currency' => $currency,
            'item_count' => count($items)
        ];
    }

    public function removeFromCart($courseId) {
        if (!$this->userId) {
            return ['success' => false, 'error' => 'Authentication required'];
        }

        $stmt = $this->db->prepare("DELETE FROM cart_items WHERE user_id = ? AND item_id = ? AND item_type = 'course'");
        $stmt->execute([$this->userId, $courseId]);

        return ['success' => true, 'message' => 'Course removed from cart'];
    }

    public function clearCart() {
        if (!$this->userId) {
            return ['success' => false, 'error' => 'Authentication required'];
        }

        $stmt = $this->db->prepare("DELETE FROM cart_items WHERE user_id = ? AND item_type = 'course'");
        $stmt->execute([$this->userId]);

        return ['success' => true, 'message' => 'Cart cleared'];
    }

    public function checkout() {
        if (!$this->userId) {
            return ['success' => false, 'error' => 'Authentication required', 'auth_required' => true];
        }

        $cart = $this->getCart();

        if (empty($cart['items'])) {
            return ['success' => false, 'error' => 'Cart is empty'];
        }

        // Separate free and paid items
        $freeItems = [];
        $paidItems = [];

        foreach ($cart['items'] as $item) {
            if ($item['is_free']) {
                $freeItems[] = $item;
            } else {
                $paidItems[] = $item;
            }
        }

        // If cart total is $0.00 (all free items), enroll directly
        if ($cart['total'] == 0.00) {
            return $this->processFreeEnrollment($freeItems);
        }

        // If cart has paid items, return payment details
        return [
            'success' => true,
            'payment_required' => true,
            'cart' => $cart,
            'free_items' => $freeItems,
            'paid_items' => $paidItems
        ];
    }

    private function processFreeEnrollment($freeItems) {
        $enrolledCourses = [];
        $errors = [];

        foreach ($freeItems as $item) {
            try {
                // Check enrollment limits
                if ($item['max_students'] > 0) {
                    $stmt = $this->db->prepare("SELECT COUNT(*) FROM course_enrollments WHERE course_id = ? AND status != 'dropped'");
                    $stmt->execute([$item['item_id']]);
                    $enrollmentCount = $stmt->fetchColumn();

                    if ($enrollmentCount >= $item['max_students']) {
                        $errors[] = "Course '{$item['title']}' is full";
                        continue;
                    }
                }

                // Create enrollment
                $stmt = $this->db->prepare("INSERT INTO course_enrollments (course_id, student_id, payment_status, payment_amount, status, enrollment_date, created_at) VALUES (?, ?, 'paid', 0.00, 'active', NOW(), NOW())");
                $stmt->execute([$item['item_id'], $this->userId]);

                $enrolledCourses[] = $item['title'];

                // Log activity
                if (function_exists('logUserActivity')) {
                    logUserActivity($this->userId, 'course_enrollment', "Enrolled in course ID: {$item['item_id']}");
                }

            } catch (Exception $e) {
                $errors[] = "Failed to enroll in '{$item['title']}': " . $e->getMessage();
            }
        }

        // Clear cart after successful enrollment
        if (!empty($enrolledCourses)) {
            $this->clearCart();
        }

        return [
            'success' => true,
            'enrolled_courses' => $enrolledCourses,
            'errors' => $errors,
            'message' => count($enrolledCourses) . ' course(s) enrolled successfully'
        ];
    }
}

$db = getDB();

// Check if user is logged in
if (!isLoggedIn()) {
    header('Location: login.php?redirect=checkout.php');
    exit;
}

$userId = $_SESSION['user_id'];
$user = getUserContext();

// Get cart contents directly
$cartManager = new CartManager($db, $userId);
$cart = $cartManager->getCart();

$page_title = 'Checkout';
?>

<!-- Main Application Container -->
<div id="app" class="flex">

    <!-- Sidebar Navigation (Desktop View) -->
    <aside class="hidden lg:block w-64 bg-white border-r border-accent-subtle p-6 shadow-sm sticky top-16 h-screen">
        <div class="flex flex-col h-full">
            <!-- Logo/Title -->
            <div class="text-xl font-extrabold text-primary-blue mb-10 tracking-wider">
            </div>

            <!-- Navigation Links -->
            <nav class="space-y-2 flex-grow">
                <a href="dashboard.php" class="flex items-center space-x-3 p-3 rounded-lg text-text-dark hover:bg-accent-subtle transition duration-150">
                    <!-- Icon Placeholder: Home -->
                    <span class="w-5 h-5">🏠</span>
                    <span>Dashboard</span>
                </a>
                <a href="available_courses.php" class="flex items-center space-x-3 p-3 rounded-lg text-text-dark hover:bg-accent-subtle transition duration-150">
                    <!-- Icon Placeholder: Catalog -->
                    <span class="w-5 h-5">📖</span>
                    <span>View All Courses</span>
                </a>
                <a href="courses.php" class="flex items-center space-x-3 p-3 rounded-lg text-text-dark hover:bg-accent-subtle transition duration-150">
                    <!-- Icon Placeholder: Book -->
                    <span class="w-5 h-5">📚</span>
                    <span>My Courses</span>
                </a>
                <a href="exams.php" class="flex items-center space-x-3 p-3 rounded-lg text-text-dark hover:bg-accent-subtle transition duration-150">
                    <!-- Icon Placeholder: Exam -->
                    <span class="w-5 h-5">📝</span>
                    <span>My Exam</span>
                </a>
                <a href="reports.php" class="flex items-center space-x-3 p-3 rounded-lg text-text-dark hover:bg-accent-subtle transition duration-150">
                    <!-- Icon Placeholder: Checkmark -->
                    <span class="w-5 h-5">✅</span>
                    <span>Grades & Reports</span>
                </a>
                <a href="calendar.php" class="flex items-center space-x-3 p-3 rounded-lg text-text-dark hover:bg-accent-subtle transition duration-150">
                    <!-- Icon Placeholder: Calendar -->
                    <span class="w-5 h-5">📅</span>
                    <span>Calendar</span>
                </a>
                <a href="../index.php" class="flex items-center space-x-3 p-3 rounded-lg text-text-dark hover:bg-accent-subtle transition duration-150">
                    <!-- Icon Placeholder: Globe -->
                    <span class="w-5 h-5">🌐</span>
                    <span>View Site</span>
                </a>
                <a href="../logout.php" class="flex items-center space-x-3 p-3 rounded-lg text-text-dark hover:bg-accent-subtle transition duration-150">
                    <!-- Icon Placeholder: Logout -->
                    <span class="w-5 h-5">🚪</span>
                    <span>Logout</span>
                </a>
            </nav>

            <!-- Footer/User Info (Bottom of Sidebar) -->
            <div class="mt-8 pt-4 border-t border-accent-subtle">
                <p class="text-xs text-gray-500 mb-2">Authenticated User:</p>
                <p class="text-xs text-gray-700 truncate font-mono bg-accent-subtle p-1 rounded"><?php echo htmlspecialchars($user['username'] ?? 'User'); ?></p>
            </div>
        </div>
    </aside>

    <!-- Main Content Area -->
    <main class="flex-1 overflow-y-auto pt-16">

    <!-- Top Bar (Mobile/Desktop Header) -->
    <header class="bg-white shadow-sm fixed top-0 left-0 right-0 z-10 border-b border-accent-subtle">
        <div class="px-4 sm:px-6 lg:px-8 py-4 flex justify-between items-center">
            <div class="flex items-center space-x-3">
                <img src="../assets/images/logo_1757657555.jpg" alt="Mutalex Academy Logo" class="h-10 w-10">
                <div>
                    <h1 class="text-xl font-bold text-primary-blue">Mutalex Academy</h1>
                    <p class="text-sm text-gray-600">Checkout</p>
                </div>
            </div>

            <div class="flex items-center space-x-4">
                <!-- Cart Button -->
                <button onclick="showDashboardCheckoutModal()" class="text-gray-600 hover:text-primary-blue p-2 rounded-full relative transition duration-150">
                    <!-- Shopping Cart Icon -->
                    <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="lucide lucide-shopping-cart">
                        <circle cx="8" cy="21" r="1"/>
                        <circle cx="19" cy="21" r="1"/>
                        <path d="M2.05 2.05h2l2.66 12.42a2 2 0 0 0 2 1.58h9.78a2 2 0 0 0 1.95-1.57l1.65-7.43H5.15"/>
                    </svg>
                    <span id="dashboard-cart-count" class="absolute -top-1 -right-1 bg-red-500 text-white text-xs font-bold rounded-full h-5 w-5 flex items-center justify-center hidden">0</span>
                </button>

                <!-- User Profile -->
                <div class="flex items-center space-x-2">
                    <span class="text-sm font-medium hidden sm:inline"><?php echo htmlspecialchars($user['username'] ?? 'Student'); ?></span>
                    <img class="h-10 w-10 rounded-full border-2 border-primary-blue object-cover" src="https://placehold.co/100x100/1E3A8A/ffffff?text=<?php echo substr(htmlspecialchars($user['username'] ?? 'S'), 0, 1); ?>" alt="User Avatar">
                    <!-- Mobile Menu Button (Hamburger) -->
                    <button class="lg:hidden p-2 rounded-lg text-text-dark hover:bg-accent-subtle ml-4" id="mobileMenuButton">
                        <span class="text-xl">☰</span>
                    </button>
                </div>
            </div>
        </div>
    </header>

        <!-- Mobile Navigation Overlay (Hidden by default) -->
        <div id="mobileMenu" class="fixed inset-0 bg-white z-20 p-6 transform -translate-x-full transition-transform duration-300 lg:hidden">
            <div class="flex justify-between items-center mb-10">
                <div class="text-xl font-extrabold text-primary-blue tracking-wider"></div>
                <button id="closeMobileMenu" class="text-3xl text-text-dark">&times;</button>
            </div>
            <nav class="space-y-4">
                <!-- Links repeated from sidebar -->
                <a href="dashboard.php" class="block p-3 rounded-lg text-text-dark hover:bg-accent-subtle">🏠 Dashboard</a>
                <a href="available_courses.php" class="block p-3 rounded-lg text-text-dark hover:bg-accent-subtle">📖 View All Courses</a>
                <a href="courses.php" class="block p-3 rounded-lg text-text-dark hover:bg-accent-subtle"> My Courses</a>
                <a href="exams.php" class="block p-3 rounded-lg text-text-dark hover:bg-accent-subtle">📝 My Exam</a>
                <a href="reports.php" class="block p-3 rounded-lg text-text-dark hover:bg-accent-subtle">✅ Grades & Reports</a>
                <a href="calendar.php" class="block p-3 rounded-lg text-text-dark hover:bg-accent-subtle">📅 Calendar</a>
                <a href="../index.php" class="block p-3 rounded-lg text-text-dark hover:bg-accent-subtle">🌐 View Site</a>
                <a href="../logout.php" class="block p-3 rounded-lg text-text-dark hover:bg-accent-subtle">🚪 Logout</a>
            </nav>
        </div>

<!-- Modern Checkout Page -->
<div class="checkout-container">
    <!-- Progress Bar -->
    <div class="checkout-progress">
        <div class="progress-bar">
            <div class="progress-step active">
                <div class="step-icon"><i class="fas fa-shopping-cart"></i></div>
                <span>Cart</span>
            </div>
            <div class="progress-line"></div>
            <div class="progress-step active">
                <div class="step-icon"><i class="fas fa-credit-card"></i></div>
                <span>Checkout</span>
            </div>
            <div class="progress-line"></div>
            <div class="progress-step">
                <div class="step-icon"><i class="fas fa-check"></i></div>
                <span>Complete</span>
            </div>
        </div>
    </div>

    <div class="checkout-main">
        <?php if (empty($cart['items'])): ?>
            <!-- Empty Cart State -->
            <div class="empty-cart">
                <div class="empty-cart-icon">
                    <i class="fas fa-shopping-cart"></i>
                </div>
                <h2>Your cart is empty</h2>
                <p>Add some courses to get started on your learning journey!</p>
                <a href="../courses/catalog.php" class="btn-primary-large">
                    <i class="fas fa-arrow-left me-2"></i>Browse Courses
                </a>
            </div>
        <?php else: ?>
            <div class="checkout-grid">
                <!-- Left Column: Order Summary -->
                <div class="order-summary">
                    <div class="summary-card">
                        <div class="summary-header">
                            <h3><i class="fas fa-receipt me-2"></i>Order Summary</h3>
                        </div>
                        <div class="summary-items">
                            <?php foreach ($cart['items'] as $item): ?>
                                <div class="summary-item">
                                    <div class="item-info">
                                        <?php if ($item['thumbnail']): ?>
                                            <img src="<?php echo htmlspecialchars($item['thumbnail']); ?>" alt="" class="item-thumbnail">
                                        <?php endif; ?>
                                        <div class="item-details">
                                            <h4><?php echo htmlspecialchars($item['title']); ?></h4>
                                            <?php if ($item['is_free']): ?>
                                                <span class="badge-free">Free</span>
                                            <?php else: ?>
                                                <span class="item-price"><?php echo formatCurrency($item['price'], $item['currency']); ?></span>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                    <button class="btn-remove" onclick="removeFromCart(<?php echo $item['item_id']; ?>)">
                                        <i class="fas fa-trash"></i>
                                    </button>
                                </div>
                            <?php endforeach; ?>
                        </div>
                        <div class="summary-total">
                            <div class="total-row">
                                <span>Total</span>
                                <span class="total-amount"><?php echo formatCurrency($cart['total'], $cart['currency']); ?></span>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Right Column: Payment Form -->
                <div class="payment-section">
                    <?php if ($cart['total'] == 0.00): ?>
                        <!-- Free Enrollment -->
                        <div class="payment-card">
                            <div class="payment-header">
                                <h3><i class="fas fa-graduation-cap me-2"></i>Free Enrollment</h3>
                            </div>
                            <div class="free-notice">
                                <i class="fas fa-check-circle"></i>
                                <p>All items in your cart are free! Click below to enroll immediately.</p>
                            </div>
                            <button id="enrollNowBtn" class="btn-primary-large" onclick="processFreeEnrollment()">
                                <i class="fas fa-graduation-cap me-2"></i>Enroll Now
                            </button>
                        </div>
                    <?php else: ?>
                        <!-- Paid Checkout -->
                        <div class="payment-card">
                            <div class="payment-header">
                                <h3><i class="fas fa-credit-card me-2"></i>Payment Details</h3>
                            </div>

                            <form id="paymentForm">
                                <input type="hidden" name="item_type" value="cart">
                                <input type="hidden" name="cart_total" value="<?php echo $cart['total']; ?>">
                                <input type="hidden" name="currency" value="<?php echo $cart['currency']; ?>">

                                <!-- Discount Code -->
                                <div class="form-group">
                                    <label>Discount Code <span class="optional">(Optional)</span></label>
                                    <input type="text" class="form-input" name="discount_code" placeholder="Enter discount code">
                                    <div id="discountMessage" class="form-hint"></div>
                                </div>

                                <!-- Payment Method -->
                                <div class="form-group">
                                    <label>Payment Method</label>
                                    <div class="payment-methods">
                                        <label class="payment-option">
                                            <input type="radio" name="gateway" value="paynow" checked>
                                            <div class="payment-content">
                                                <i class="fas fa-mobile-alt"></i>
                                                <span>Paynow (Mobile Money)</span>
                                            </div>
                                        </label>
                                        <label class="payment-option">
                                            <input type="radio" name="gateway" value="paypal">
                                            <div class="payment-content">
                                                <i class="fab fa-paypal"></i>
                                                <span>PayPal</span>
                                            </div>
                                        </label>
                                    </div>
                                </div>

                                <!-- Credit Card Form (for PayPal) -->
                                <div id="creditCardForm" class="credit-card-section" style="display: none;">
                                    <h4>Credit Card Information</h4>
                                    <div class="card-form">
                                        <div class="form-group">
                                            <label>Card Number</label>
                                            <input type="text" class="form-input" name="card_number" placeholder="1234 5678 9012 3456">
                                        </div>
                                        <div class="form-row">
                                            <div class="form-group">
                                                <label>Expiry Date</label>
                                                <input type="text" class="form-input" name="expiry_date" placeholder="MM/YY">
                                            </div>
                                            <div class="form-group">
                                                <label>CVV</label>
                                                <input type="text" class="form-input" name="cvv" placeholder="123">
                                            </div>
                                        </div>
                                        <div class="form-group">
                                            <label>Cardholder Name</label>
                                            <input type="text" class="form-input" name="cardholder_name" placeholder="John Doe">
                                        </div>
                                    </div>
                                </div>

                                <!-- Final Amount -->
                                <div id="finalAmount" class="final-amount">
                                    <div class="amount-display">
                                        <span>Total Amount</span>
                                        <span class="amount"><?php echo formatCurrency($cart['total'], $cart['currency']); ?></span>
                                    </div>
                                </div>

                                <!-- PayNow Button (shown when PayNow is selected) -->
                                <div id="paynowSection" style="display: none; text-align: center; margin-bottom: 1.5rem;">
                                    <a href='https://www.paynow.co.zw/Payment/Link/?q=c2VhcmNoPW11dGFsZXh6dyU0MGdtYWlsLmNvbSZhbW91bnQ9MS4wMCZyZWZlcmVuY2U9TUEwMDAwMDAmbD0x' target='_blank'>
                                        <img src='https://www.paynow.co.zw/Content/Buttons/Medium_buttons/button_pay-now_medium.png' style='border:0' />
                                    </a>
                                </div>

                                <!-- Pay Button -->
                                <button type="submit" class="btn-primary-large" id="payButton">
                                    <i class="fas fa-lock me-2"></i>
                                    Pay <?php echo formatCurrency($cart['total'], $cart['currency']); ?>
                                </button>
                            </form>
                        </div>
                    <?php endif; ?>

                    <!-- Trust Badges -->
                    <div class="trust-badges">
                        <div class="badge">
                            <i class="fas fa-shield-alt"></i>
                            <span>Secure Payment</span>
                        </div>
                        <div class="badge">
                            <i class="fas fa-lock"></i>
                            <span>SSL Encrypted</span>
                        </div>
                        <div class="badge">
                            <i class="fas fa-undo"></i>
                            <span>Money Back Guarantee</span>
                        </div>
                    </div>
                </div>
            </div>
        <?php endif; ?>
    </div>
</div>

<script>
// Initialize cart on page load
document.addEventListener('DOMContentLoaded', function() {
    loadCartContent();
});

// Load cart content
function loadCartContent() {
    fetch('../api/cart.php')
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            // Update cart count badge
            const cartBadge = document.getElementById('dashboard-cart-count');
            if (cartBadge) {
                const count = data.cart.item_count || 0;
                cartBadge.textContent = count;
                cartBadge.classList.toggle('hidden', count === 0);
            }

            if (data.cart.items.length > 0) {
                // Cart has items, checkout modal can be shown when needed
                console.log('Cart loaded:', data.cart);
            }
        }
    })
    .catch(error => console.error('Error loading cart:', error));
}

// Show checkout modal from dashboard
function showDashboardCheckoutModal() {
    // Create modal HTML
    const modalHtml = `
        <div id="dashboard-checkout-modal" class="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
            <div class="bg-white rounded-xl shadow-2xl max-w-md w-full mx-4 max-h-[90vh] overflow-y-auto">
                <div class="p-6">
                    <div class="flex justify-between items-center mb-6">
                        <h2 class="text-xl font-bold text-gray-900">Complete Checkout</h2>
                        <button onclick="closeDashboardCheckoutModal()" class="text-gray-400 hover:text-gray-600">
                            <i class="fas fa-times text-xl"></i>
                        </button>
                    </div>

                    <div id="checkout-content">
                        <!-- Content will be loaded here -->
                        <div class="text-center py-8">
                            <div class="animate-spin rounded-full h-8 w-8 border-b-2 border-primary-blue mx-auto mb-4"></div>
                            <p class="text-gray-600">Loading checkout...</p>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    `;

    document.body.insertAdjacentHTML('beforeend', modalHtml);

    // Add event listeners for closing modal
    const modal = document.getElementById('dashboard-checkout-modal');

    // Close modal when clicking outside
    modal.addEventListener('click', function(e) {
        if (e.target === modal) {
            closeDashboardCheckoutModal();
        }
    });

    // Close modal when pressing ESC key
    const handleEscapeKey = function(e) {
        if (e.key === 'Escape') {
            closeDashboardCheckoutModal();
        }
    };
    document.addEventListener('keydown', handleEscapeKey);

    // Store the escape handler to remove it later
    modal._escapeHandler = handleEscapeKey;

    loadCheckoutContent();
}

// Close checkout modal
function closeDashboardCheckoutModal() {
    const modal = document.getElementById('dashboard-checkout-modal');
    if (modal) {
        // Remove escape key event listener
        if (modal._escapeHandler) {
            document.removeEventListener('keydown', modal._escapeHandler);
        }
        modal.remove();
    }
}

// Load checkout content
function loadCheckoutContent() {
    fetch('../api/cart.php')
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            const cart = data.cart;
            renderCheckoutContent(cart);
        } else {
            document.getElementById('checkout-content').innerHTML = '<div class="text-center py-8"><p class="text-red-600">Error loading cart</p></div>';
        }
    })
    .catch(error => {
        console.error('Error loading checkout:', error);
        document.getElementById('checkout-content').innerHTML = '<div class="text-center py-8"><p class="text-red-600">Error loading checkout</p></div>';
    });
}

// Render checkout content
function renderCheckoutContent(cart) {
    if (!cart.items || cart.items.length === 0) {
        document.getElementById('checkout-content').innerHTML = `
            <div class="text-center py-8">
                <i class="fas fa-shopping-cart fa-3x text-gray-300 mb-4"></i>
                <h5 class="text-gray-600 mb-2">Your cart is empty</h5>
                <p class="text-sm text-gray-500 mb-6">Add some courses to get started with your learning journey</p>
                <div class="flex flex-col sm:flex-row gap-3 justify-center">
                    <button onclick="closeDashboardCheckoutModal(); toggleAvailableCourses();" class="bg-primary text-white px-6 py-2 rounded-lg hover:bg-indigo-700 font-medium">
                        <i class="fas fa-graduation-cap mr-2"></i>Browse Courses
                    </button>
                    <a href="../courses/catalog.php" class="bg-white text-primary border border-primary px-6 py-2 rounded-lg hover:bg-blue-50 font-medium">
                        <i class="fas fa-external-link-alt mr-2"></i>View Full Catalog
                    </a>
                </div>
            </div>
        `;
        return;
    }

    let itemsHtml = '';
    cart.items.forEach(item => {
        itemsHtml += `
            <div class="bg-gray-50 rounded-lg p-4 mb-3 border border-gray-200">
                <div class="flex items-start justify-between">
                    <div class="flex items-start space-x-3 flex-1">
                        <div class="flex-1">
                            <h4 class="text-sm font-medium text-gray-800">${item.title}</h4>
                            <p class="text-xs text-gray-500">${item.is_free ? 'Free' : formatCurrency(item.price, item.currency)}</p>
                        </div>
                    </div>
                </div>
            </div>
        `;
    });

    const checkoutHtml = `
        <div class="mb-6">
            <div class="flex justify-between items-center mb-4">
                <h3 class="font-medium text-gray-800">Cart Items (${cart.items.length})</h3>
                <div class="flex items-center space-x-3">
                    <button onclick="closeDashboardCheckoutModal(); toggleAvailableCourses();" class="text-sm text-blue-600 hover:text-blue-800 font-medium flex items-center">
                        <i class="fas fa-plus mr-1"></i>Add More Courses
                    </button>
                    <button onclick="clearCart()" class="text-xs text-gray-500 hover:text-red-600 font-medium flex items-center ml-4 px-2 py-1 border border-gray-300 rounded">
                        <i class="fas fa-trash mr-1"></i>Clear All
                    </button>
                </div>
            </div>
            ${itemsHtml}
        </div>

        <div class="bg-gray-50 p-4 rounded-lg mb-6">
            <div class="flex justify-between items-center">
                <span class="font-medium text-gray-700">Total Amount:</span>
                <span class="text-2xl font-bold text-primary">${formatCurrency(cart.total, cart.currency)}</span>
            </div>
        </div>

        <div class="text-center space-y-3">
            <button onclick="processFreeEnrollment()" class="w-full bg-green-600 text-white py-3 px-4 rounded-lg font-semibold hover:bg-green-700 transition duration-200">
                <i class="fas fa-graduation-cap mr-2"></i>Enroll Now
            </button>
        </div>
    `;

    document.getElementById('checkout-content').innerHTML = checkoutHtml;
}

// Format currency for JavaScript
function formatCurrency(amount, currency = 'USD') {
    const formatter = new Intl.NumberFormat('en-US', {
        style: 'currency',
        currency: currency,
        minimumFractionDigits: 2,
        maximumFractionDigits: 2
    });
    return formatter.format(amount);
}

// Clear entire cart
function clearCart() {
    if (confirm('Are you sure you want to clear your entire cart?')) {
        fetch('../api/cart.php', {
            method: 'DELETE'
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showToast('Cart cleared successfully', 'success');
                loadCheckoutContent(); // Reload the checkout modal
                loadCartContent(); // Update cart count badge
            } else {
                showToast('Error clearing cart: ' + data.error, 'error');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            showToast('An error occurred. Please try again.', 'error');
        });
    }
}

// Process free enrollment from dashboard
function processFreeEnrollment() {
    fetch('../api/cart.php', {
        method: 'PUT',
        headers: {
            'Content-Type': 'application/json',
        }
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showToast('Successfully enrolled in courses!', 'success');
            closeDashboardCheckoutModal();
            loadCartContent(); // Reload cart count
            setTimeout(() => {
                location.reload(); // Reload page to show enrolled courses
            }, 1500);
        } else {
            showToast('Error: ' + (data.error || 'Enrollment failed'), 'error');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showToast('An error occurred. Please try again.', 'error');
    });
}

// Mobile Menu Functionality
const mobileMenuButton = document.getElementById('mobileMenuButton');
const closeMobileMenu = document.getElementById('closeMobileMenu');
const mobileMenu = document.getElementById('mobileMenu');

if (mobileMenuButton && closeMobileMenu && mobileMenu) {
    mobileMenuButton.addEventListener('click', () => {
        mobileMenu.classList.remove('-translate-x-full');
    });

    closeMobileMenu.addEventListener('click', () => {
        mobileMenu.classList.add('-translate-x-full');
    });

    // Close menu when a link is clicked
    mobileMenu.querySelectorAll('a').forEach(link => {
        link.addEventListener('click', () => {
            mobileMenu.classList.add('-translate-x-full');
        });
    });
}

// Toast notification system
function showToast(message, type = 'success') {
    // Remove existing toasts
    const existingToasts = document.querySelectorAll('.toast-notification');
    existingToasts.forEach(toast => toast.remove());

    // Create toast element
    const toast = document.createElement('div');
    toast.className = `toast-notification fixed top-4 right-4 z-50 p-4 rounded-lg shadow-lg transform translate-x-full transition-all duration-300 ${
        type === 'success' ? 'bg-green-500 text-white' : 'bg-red-500 text-white'
    }`;
    toast.innerHTML = `
        <div class="flex items-center gap-3">
            <i class="fas ${type === 'success' ? 'fa-check-circle' : 'fa-exclamation-triangle'}"></i>
            <span>${message}</span>
        </div>
    `;

    // Add to page
    document.body.appendChild(toast);

    // Animate in
    setTimeout(() => {
        toast.classList.remove('translate-x-full');
    }, 100);

    // Auto remove after 4 seconds
    setTimeout(() => {
        toast.classList.add('translate-x-full');
        setTimeout(() => {
            toast.remove();
        }, 300);
    }, 4000);
}

// Toggle forms based on payment method
document.querySelectorAll('input[name="gateway"]').forEach(radio => {
    radio.addEventListener('change', function() {
        const creditCardForm = document.getElementById('creditCardForm');
        const paynowSection = document.getElementById('paynowSection');
        if (this.value === 'paypal') {
            creditCardForm.style.display = 'block';
            paynowSection.style.display = 'none';
        } else if (this.value === 'paynow') {
            creditCardForm.style.display = 'none';
            paynowSection.style.display = 'block';
        } else {
            creditCardForm.style.display = 'none';
            paynowSection.style.display = 'none';
        }
    });
});

// Handle discount code validation
document.querySelector('input[name="discount_code"]')?.addEventListener('blur', function() {
    validateDiscountCode(this.value, <?php echo $cart['total']; ?>, '<?php echo $cart['currency']; ?>');
});

function validateDiscountCode(code, originalPrice, currency) {
    if (!code.trim()) {
        document.getElementById('discountMessage').innerHTML = '';
        updateFinalAmount(originalPrice, 0, currency);
        return;
    }

    document.getElementById('discountMessage').innerHTML = '<small class="text-muted">Validating discount code...</small>';

    // Simulate API call
    setTimeout(() => {
        if (code.toUpperCase().startsWith('TEST')) {
            const discountAmount = originalPrice * 0.1;
            document.getElementById('discountMessage').innerHTML = '<small class="text-success">10% discount applied!</small>';
            updateFinalAmount(originalPrice, discountAmount, currency);
        } else {
            document.getElementById('discountMessage').innerHTML = '<small class="text-danger">Invalid discount code</small>';
            updateFinalAmount(originalPrice, 0, currency);
        }
    }, 500);
}

function updateFinalAmount(originalPrice, discountAmount, currency) {
    const finalAmount = originalPrice - discountAmount;
    document.getElementById('finalAmount').innerHTML = `<strong>Final Amount: ${currency} ${finalAmount.toFixed(2)}</strong>`;
    document.getElementById('payButton').innerHTML = `<i class="fas fa-credit-card me-2"></i>Pay ${currency} ${finalAmount.toFixed(2)}`;
}

function removeFromCart(courseId) {
    if (confirm('Remove this course from cart?')) {
        fetch('../api/cart.php?course_id=' + courseId, {
            method: 'DELETE'
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                location.reload();
            } else {
                alert('Error: ' + data.error);
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('An error occurred. Please try again.');
        });
    }
}

function processFreeEnrollment() {
    const btn = document.getElementById('enrollNowBtn');
    const originalText = btn.innerHTML;

    btn.disabled = true;
    btn.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>Enrolling...';

    fetch('../api/cart.php', {
        method: 'PUT',
        headers: {
            'Content-Type': 'application/json',
        }
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            alert('Successfully enrolled in ' + data.enrolled_courses.length + ' course(s)!');
            window.location.href = '../profile.php';
        } else {
            alert('Error: ' + (data.error || 'Enrollment failed'));
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('An error occurred. Please try again.');
    })
    .finally(() => {
        btn.disabled = false;
        btn.innerHTML = originalText;
    });
}

// Handle payment form submission
document.getElementById('paymentForm')?.addEventListener('submit', function(e) {
    e.preventDefault();
    processCartPayment(new FormData(this));
});

function processCartPayment(formData) {
    const payButton = document.getElementById('payButton');
    const originalText = payButton.innerHTML;

    payButton.disabled = true;
    payButton.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>Processing...';

    fetch('../api/payments.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            if (data.redirect_url) {
                window.location.href = data.redirect_url;
            } else {
                alert('Payment initiated successfully!');
                window.location.href = 'dashboard.php';
            }
        } else {
            showPaymentError(data.error || 'Unknown error');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('An error occurred. Please try again.');
    })
    .finally(() => {
        payButton.disabled = false;
        payButton.innerHTML = originalText;
    });
}
</script>

<style>
/* Modern Checkout Styles */
.checkout-container {
    max-width: 1200px;
    margin: 0 auto;
    padding: 2rem 1rem;
    background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
    min-height: calc(100vh - 160px);
}

.checkout-progress {
    margin-bottom: 2rem;
}

.progress-bar {
    display: flex;
    align-items: center;
    justify-content: center;
    max-width: 600px;
    margin: 0 auto;
}

.progress-step {
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 0.5rem;
    opacity: 0.5;
    transition: all 0.3s ease;
}

.progress-step.active {
    opacity: 1;
}

.step-icon {
    width: 50px;
    height: 50px;
    border-radius: 50%;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-size: 1.2rem;
    box-shadow: 0 4px 15px rgba(102, 126, 234, 0.3);
}

.progress-step.active .step-icon {
    background: linear-gradient(135deg, #4f46e5 0%, #7c3aed 100%);
    animation: pulse 2s infinite;
}

.progress-line {
    width: 80px;
    height: 2px;
    background: #e5e7eb;
    margin: 0 1rem;
}

.progress-step.active + .progress-line {
    background: linear-gradient(90deg, #4f46e5, #7c3aed);
}

.checkout-main {
    max-width: 1000px;
    margin: 0 auto;
}

.checkout-grid {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 2rem;
    align-items: start;
}

/* Empty Cart */
.empty-cart {
    text-align: center;
    padding: 4rem 2rem;
    background: white;
    border-radius: 1rem;
    box-shadow: 0 10px 25px rgba(0, 0, 0, 0.1);
    grid-column: 1 / -1;
}

.empty-cart-icon {
    font-size: 4rem;
    color: #cbd5e1;
    margin-bottom: 1rem;
}

.empty-cart h2 {
    color: #1f2937;
    margin-bottom: 1rem;
    font-size: 1.8rem;
}

.empty-cart p {
    color: #64748b;
    margin-bottom: 2rem;
    font-size: 1.1rem;
}

/* Order Summary */
.order-summary {
    /* Removed sticky positioning */
}

.summary-card {
    background: white;
    border-radius: 1rem;
    box-shadow: 0 10px 25px rgba(0, 0, 0, 0.1);
    overflow: hidden;
    border: 1px solid #e5e7eb;
}

.summary-header {
    background: linear-gradient(135deg, #4f46e5 0%, #7c3aed 100%);
    color: white;
    padding: 1.5rem;
}

.summary-header h3 {
    margin: 0;
    font-size: 1.3rem;
    font-weight: 600;
}

.summary-items {
    padding: 1.5rem;
}

.summary-item {
    display: flex;
    align-items: center;
    justify-content: space-between;
    padding: 1rem 0;
    border-bottom: 1px solid #f1f5f9;
}

.summary-item:last-child {
    border-bottom: none;
}

.item-info {
    display: flex;
    align-items: center;
    gap: 1rem;
    flex: 1;
}

.item-thumbnail {
    width: 60px;
    height: 40px;
    object-fit: cover;
    border-radius: 0.5rem;
    border: 2px solid #e5e7eb;
}

.item-details h4 {
    margin: 0 0 0.25rem 0;
    font-size: 1rem;
    font-weight: 600;
    color: #1f2937;
}

.badge-free {
    background: #dcfce7;
    color: #166534;
    padding: 0.25rem 0.75rem;
    border-radius: 0.5rem;
    font-size: 0.8rem;
    font-weight: 600;
}

.item-price {
    color: #4f46e5;
    font-weight: 600;
    font-size: 1rem;
}

.btn-remove {
    background: none;
    border: none;
    color: #dc2626;
    cursor: pointer;
    padding: 0.5rem;
    border-radius: 0.5rem;
    transition: all 0.3s ease;
}

.btn-remove:hover {
    background: #fef2f2;
    transform: scale(1.1);
}

.summary-total {
    background: #f8fafc;
    padding: 1.5rem;
    border-top: 1px solid #e5e7eb;
}

.total-row {
    display: flex;
    justify-content: space-between;
    align-items: center;
    font-size: 1.2rem;
    font-weight: 600;
    color: #1f2937;
}

.total-amount {
    color: #4f46e5;
    font-size: 1.4rem;
}

/* Payment Section */
.payment-section {
    display: flex;
    flex-direction: column;
    gap: 2rem;
}

.payment-card {
    background: white;
    border-radius: 1rem;
    box-shadow: 0 10px 25px rgba(0, 0, 0, 0.1);
    overflow: hidden;
    border: 1px solid #e5e7eb;
}

.payment-header {
    background: linear-gradient(135deg, #10b981 0%, #059669 100%);
    color: white;
    padding: 1.5rem;
}

.payment-header h3 {
    margin: 0;
    font-size: 1.3rem;
    font-weight: 600;
}

.free-notice {
    padding: 2rem;
    text-align: center;
    background: #f0fdf4;
    border-bottom: 1px solid #bbf7d0;
}

.free-notice i {
    font-size: 3rem;
    color: #16a34a;
    margin-bottom: 1rem;
}

.free-notice p {
    color: #166534;
    font-size: 1.1rem;
    margin: 0;
}

/* Form Styles */
.form-group {
    margin-bottom: 1.5rem;
}

.form-group label {
    display: block;
    margin-bottom: 0.5rem;
    font-weight: 600;
    color: #374151;
    font-size: 0.95rem;
}

.optional {
    font-weight: 400;
    color: #6b7280;
    font-size: 0.85rem;
}

.form-input {
    width: 100%;
    padding: 0.875rem 1rem;
    border: 2px solid #e5e7eb;
    border-radius: 0.5rem;
    font-size: 1rem;
    transition: all 0.3s ease;
    background: white;
}

.form-input:focus {
    outline: none;
    border-color: #4f46e5;
    box-shadow: 0 0 0 3px rgba(79, 70, 229, 0.1);
}

.form-hint {
    margin-top: 0.25rem;
    font-size: 0.85rem;
}

.form-row {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 1rem;
}

/* Payment Methods */
.payment-methods {
    display: flex;
    flex-direction: column;
    gap: 1rem;
}

.payment-option {
    display: flex;
    align-items: center;
    cursor: pointer;
    padding: 1rem;
    border: 2px solid #e5e7eb;
    border-radius: 0.5rem;
    transition: all 0.3s ease;
    background: white;
}

.payment-option:hover {
    border-color: #4f46e5;
    box-shadow: 0 4px 12px rgba(79, 70, 229, 0.1);
}

.payment-option input[type="radio"] {
    margin-right: 1rem;
    accent-color: #4f46e5;
}

.payment-content {
    display: flex;
    align-items: center;
    gap: 0.75rem;
    flex: 1;
}

.payment-content i {
    font-size: 1.5rem;
    color: #4f46e5;
}

.payment-content span {
    font-weight: 600;
    color: #1f2937;
}

/* Credit Card Form */
.credit-card-section {
    padding: 0 1.5rem 1.5rem;
}

.credit-card-section h4 {
    margin-bottom: 1rem;
    color: #1f2937;
    font-size: 1.1rem;
}

.card-form {
    background: #f8fafc;
    padding: 1.5rem;
    border-radius: 0.5rem;
    border: 1px solid #e5e7eb;
}

/* Final Amount */
.final-amount {
    padding: 1.5rem;
    background: #f0f9ff;
    border: 1px solid #bae6fd;
    border-radius: 0.5rem;
    margin-bottom: 1.5rem;
}

.amount-display {
    display: flex;
    justify-content: space-between;
    align-items: center;
    font-size: 1.2rem;
    font-weight: 600;
    color: #1f2937;
}

.amount {
    color: #4f46e5;
    font-size: 1.4rem;
}

/* Buttons */
.btn-primary-large {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    gap: 0.5rem;
    width: 100%;
    padding: 1rem 2rem;
    background: linear-gradient(135deg, #4f46e5 0%, #7c3aed 100%);
    color: white;
    border: none;
    border-radius: 0.5rem;
    font-size: 1.1rem;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s ease;
    text-decoration: none;
    box-shadow: 0 4px 15px rgba(79, 70, 229, 0.3);
}

.btn-primary-large:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(79, 70, 229, 0.4);
}

.btn-primary-large:active {
    transform: translateY(0);
}

/* Trust Badges */
.trust-badges {
    display: flex;
    justify-content: center;
    gap: 2rem;
    margin-top: 2rem;
}

.badge {
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 0.5rem;
    padding: 1rem;
    background: white;
    border-radius: 0.5rem;
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
    border: 1px solid #e5e7eb;
    min-width: 120px;
}

.badge i {
    font-size: 1.5rem;
    color: #10b981;
}

.badge span {
    font-size: 0.85rem;
    font-weight: 600;
    color: #374151;
    text-align: center;
}

/* Animations */
@keyframes pulse {
    0%, 100% { transform: scale(1); }
    50% { transform: scale(1.05); }
}

@keyframes fadeInUp {
    from {
        opacity: 0;
        transform: translateY(30px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.checkout-container {
    animation: fadeInUp 0.6s ease-out;
}

/* Responsive Design */
@media (max-width: 768px) {
    .checkout-container {
        padding: 1rem;
    }

    .checkout-grid {
        grid-template-columns: 1fr;
        gap: 1.5rem;
    }

    .order-summary {
        position: static;
    }

    .progress-bar {
        flex-direction: column;
        gap: 1rem;
    }

    .progress-line {
        width: 2px;
        height: 40px;
        margin: 0;
    }

    .trust-badges {
        flex-direction: column;
        gap: 1rem;
    }

    .badge {
        flex-direction: row;
        min-width: auto;
    }

    .form-row {
        grid-template-columns: 1fr;
    }

    .checkout-progress {
        margin-bottom: 1.5rem;
    }
}

@media (max-width: 480px) {
    .checkout-container {
        padding: 0.5rem;
    }

    .empty-cart {
        padding: 2rem 1rem;
    }

    .empty-cart-icon {
        font-size: 3rem;
    }

    .summary-items,
    .summary-total {
        padding: 1rem;
    }

    .payment-card {
        margin: 0 -0.5rem;
    }
}
</style>

    </main>
</div>

<?php include '../includes/footer.php'; ?>